<?php
namespace DynamicHeartbeat;

defined('ABSPATH') || exit;

if (!defined('DFEHC_CACHE_GROUP')) {
    define('DFEHC_CACHE_GROUP', 'dfehc');
}

class Dfehc_ServerLoadEstimator {
    const BASELINE_TRANSIENT_PREFIX = 'dfehc_baseline_';
    const LOAD_CACHE_TRANSIENT      = 'dfehc_last_known_load';
    const LOAD_SPIKE_TRANSIENT      = 'dfehc_load_spike_score';
    const BASELINE_RESET_CD_PREFIX  = 'dfehc_baseline_reset_cd_';

    public static function get_server_load(float $duration = 0.025) {
        if (\apply_filters('dfehc_disable_loop_estimator', false)) {
            return false;
        }
        if (!\function_exists('microtime') || (\defined('DFEHC_DISABLE_LOAD_ESTIMATION') && DFEHC_DISABLE_LOAD_ESTIMATION)) {
            return false;
        }
        $duration = (float) \apply_filters('dfehc_loop_duration', $duration);
        if (!\is_finite($duration) || $duration <= 0.0) {
            $duration = 0.025;
        }
        if ($duration < 0.01) {
            $duration = 0.01;
        }
        if ($duration > 0.5) {
            $duration = 0.5;
        }
        $suffix    = self::scope_suffix();
        $baselineT = self::get_baseline_transient_name($suffix);
        $cacheTtl  = (int) \apply_filters('dfehc_load_cache_ttl', 90);
        $cacheKey  = self::get_cache_key($suffix);
        $cached = self::get_scoped_transient($cacheKey);
        if ($cached !== false && $cached !== null) {
            return $cached;
        }
        $sysAvg = self::try_sys_getloadavg();
        if ($sysAvg !== null) {
            self::set_scoped_transient_noautoload($cacheKey, $sysAvg, $cacheTtl);
            return $sysAvg;
        }
        $baseline = self::get_baseline_value($baselineT);
        if ($baseline === false || $baseline === null || !\is_numeric($baseline) || $baseline <= 0) {
            $baseline = self::maybe_calibrate($baselineT, $duration);
        }
        $loopsPerSec = self::run_loop_avg($duration);
        if ($loopsPerSec <= 0) {
            return false;
        }
        $loadRatio   = ($baseline * 0.125) / \max($loopsPerSec, 1);
        $loadPercent = \round(\min(100.0, \max(0.0, $loadRatio * 100.0)), 2);
        $loadPercent = (float) \apply_filters('dfehc_computed_load_percent', $loadPercent, $baseline, $loopsPerSec);
        self::update_spike_score($loadPercent, $suffix);
        self::set_scoped_transient_noautoload($cacheKey, $loadPercent, $cacheTtl);
        return $loadPercent;
    }

    public static function calibrate_baseline(float $duration = 0.025): float {
        $duration = (float) \apply_filters('dfehc_loop_duration', $duration);
        if (!\is_finite($duration) || $duration <= 0.0) {
            $duration = 0.025;
        }
        if ($duration < 0.01) {
            $duration = 0.01;
        }
        if ($duration > 0.5) {
            $duration = 0.5;
        }
        return self::run_loop_avg($duration);
    }

    public static function maybe_calibrate_if_idle(): void {
        if (\is_admin() || \is_user_logged_in()) {
            return;
        }
        if ((\function_exists('wp_doing_cron') && \wp_doing_cron()) ||
            (\function_exists('wp_doing_ajax') && \wp_doing_ajax()) ||
            (\function_exists('wp_is_json_request') && \wp_is_json_request()) ||
            (\defined('WP_CLI') && WP_CLI)) {
            return;
        }
        $suffix  = self::scope_suffix();
        $seenKey = 'dfehc_seen_recently_' . $suffix;
        $ttl     = (int) \apply_filters('dfehc_seen_recently_ttl', 60);
        if (\get_transient($seenKey) !== false) {
            return;
        }
        self::set_transient_noautoload($seenKey, 1, $ttl);
        self::ensure_baseline();
    }

    public static function maybe_calibrate_during_cron(): void {
        if (!\function_exists('wp_doing_cron') || !\wp_doing_cron()) {
            return;
        }
        $suffix = self::scope_suffix();
        $cdKey  = 'dfehc_cron_cal_cd_' . $suffix;
        $cdTtl  = (int) \apply_filters('dfehc_cron_calibration_cooldown', 300);
        if (\get_transient($cdKey) !== false) {
            return;
        }
        self::set_transient_noautoload($cdKey, 1, $cdTtl);
        self::ensure_baseline();
    }

    private static function try_sys_getloadavg(): ?float {
        if (!\function_exists('sys_getloadavg')) {
            return null;
        }
        $avg = \sys_getloadavg();
        if (!\is_array($avg) || !isset($avg[0])) {
            return null;
        }
        $raw = (float) $avg[0];
        $raw = (float) \apply_filters('dfehc_raw_sys_load', $raw);
        $cores = 0;
        if (\defined('DFEHC_CPU_CORES')) {
            $cores = (int) DFEHC_CPU_CORES;
        } elseif (\function_exists('\dfehc_get_cpu_cores')) {
            $cores = (int) \dfehc_get_cpu_cores();
        } elseif (\function_exists('dfehc_get_cpu_cores')) {
            $cores = (int) \dfehc_get_cpu_cores();
        }
        $cores = (int) \apply_filters('dfehc_cpu_cores', $cores);
        if ($cores <= 0) {
            $cores = 1;
        }
        return \min(100.0, \round(($raw / $cores) * 100.0, 2));
    }

    private static function now(): float {
        return \function_exists('hrtime') ? (hrtime(true) / 1e9) : \microtime(true);
    }

    private static function run_loop(float $duration): float {
        if ($duration <= 0.0) {
            return 0.0;
        }
        if ($duration < 0.01) {
            $duration = 0.01;
        }
        if ($duration > 0.5) {
            $duration = 0.5;
        }
        $duration += \mt_rand(0, 2) * 0.001;
        $warm = self::now();
        for ($i = 0; $i < 1000; $i++) { $warm += 0; }
        $start = self::now();
        $end   = $start + $duration;
        $cnt   = 0;
        $cap   = (int) \apply_filters('dfehc_loop_iteration_cap', 10000000);
        $now   = $start;
        while ($now < $end && $cnt < $cap) {
            ++$cnt;
            $now = self::now();
        }
        $elapsed = $now - $start;
        return $elapsed > 0 ? $cnt / $elapsed : 0.0;
    }

    private static function run_loop_avg(float $duration): float {
        $a = self::run_loop($duration);
        $b = self::run_loop(\min(0.5, $duration * 1.5));
        if ($a <= 0.0 && $b <= 0.0) {
            return 0.0;
        }
        if ($a <= 0.0) return $b;
        if ($b <= 0.0) return $a;
        return ($a + $b) / 2.0;
    }

    private static function maybe_calibrate(string $baselineT, float $duration): float {
        $suffix   = self::scope_suffix();
        $lockKey  = 'dfehc_calibrating_' . $suffix;
        $lock     = self::acquire_lock($lockKey, 30);
        $baseline = self::run_loop_avg($duration);
        if ($baseline <= 0.0) {
            $baseline = 1.0;
        }
        if ($lock) {
            $exp = (int) \apply_filters('dfehc_baseline_expiration', 7 * DAY_IN_SECONDS);
            self::set_baseline_value($baselineT, $baseline, $exp);
            self::release_lock($lock, $lockKey);
        }
        return $baseline;
    }

    private static function update_spike_score(float $loadPercent, string $suffix): void {
        $spikeKey   = self::get_spike_key($suffix);
        $score      = (float) self::get_scoped_transient($spikeKey);
        $decay      = (float) \apply_filters('dfehc_spike_decay', 0.5);
        $increment  = (float) \apply_filters('dfehc_spike_increment', 1.0);
        $threshold  = (float) \apply_filters('dfehc_spike_threshold', 3.0);
        $trigger    = (float) \apply_filters('dfehc_spike_trigger', 90.0);
        $resetCdTtl = (int) \apply_filters('dfehc_baseline_reset_cooldown', 3600);
        $resetCdKey = self::BASELINE_RESET_CD_PREFIX . $suffix;
        $scoreMax   = (float) \apply_filters('dfehc_spike_score_max', 20.0);
        if ($loadPercent > $trigger) {
            $score += $increment * (1 + (($loadPercent - $trigger) / 20.0));
        } else {
            $score = \max(0.0, $score - $decay);
        }
        $score = \min($scoreMax, \max(0.0, $score));
        if ($score >= $threshold) {
            if (\get_transient($resetCdKey) === false) {
                $baselineName = self::get_baseline_transient_name($suffix);
                self::delete_baseline_value($baselineName);
                self::set_transient_noautoload($resetCdKey, 1, $resetCdTtl);
                self::delete_scoped_transient($spikeKey);
                return;
            }
        }
        self::set_scoped_transient_noautoload($spikeKey, $score, (int) \apply_filters('dfehc_spike_score_ttl', HOUR_IN_SECONDS));
    }

    private static function ensure_baseline(): void {
        $suffix    = self::scope_suffix();
        $baselineT = self::get_baseline_transient_name($suffix);
        $existing  = self::get_baseline_value($baselineT);
        if ($existing !== false && $existing !== null && \is_numeric($existing) && $existing > 0) {
            return;
        }
        $lockKey = 'dfehc_calibrating_' . $suffix;
        $lock    = self::acquire_lock($lockKey, 30);
        if (!$lock) {
            return;
        }
        $duration = (float) \apply_filters('dfehc_loop_duration', 0.025);
        if (!\is_finite($duration) || $duration <= 0.0) {
            $duration = 0.025;
        }
        if ($duration < 0.01) {
            $duration = 0.01;
        }
        if ($duration > 0.5) {
            $duration = 0.5;
        }
        $baseline = self::run_loop_avg($duration);
        if ($baseline <= 0.0) {
            $baseline = 1.0;
        }
        $exp = (int) \apply_filters('dfehc_baseline_expiration', 7 * DAY_IN_SECONDS);
        self::set_baseline_value($baselineT, $baseline, $exp);
        self::release_lock($lock, $lockKey);
    }

    private static function acquire_lock(string $key, int $ttl) {
        if (\class_exists('\WP_Lock')) {
            $lock = new \WP_Lock($key, $ttl);
            return $lock->acquire() ? $lock : null;
        }
        if (\function_exists('wp_cache_add') && \wp_cache_add($key, 1, DFEHC_CACHE_GROUP, $ttl)) {
            return (object) ['type' => 'cache', 'key' => $key];
        }
        if (\get_transient($key) !== false) {
            return null;
        }
        if (\set_transient($key, 1, $ttl)) {
            return (object) ['type' => 'transient', 'key' => $key];
        }
        return null;
    }

    private static function release_lock($lock, string $key): void {
        if ($lock instanceof \WP_Lock) {
            $lock->release();
            return;
        }
        if (\is_object($lock) && isset($lock->type, $lock->key)) {
            if ($lock->type === 'cache') {
                \wp_cache_delete($key, DFEHC_CACHE_GROUP);
                return;
            }
            if ($lock->type === 'transient') {
                \delete_transient($key);
                return;
            }
        }
    }

    private static function get_baseline_transient_name(string $suffix): string {
        return self::BASELINE_TRANSIENT_PREFIX . $suffix;
    }

    private static function get_hostname_key(): string {
        $host = @\php_uname('n');
        if (!$host) {
            $url = \defined('WP_HOME') && WP_HOME ? WP_HOME : (\function_exists('home_url') ? \home_url() : '');
            $parts = \parse_url((string) $url);
            $host = $parts['host'] ?? ($url ?: 'unknown');
        }
        $salt = \defined('DB_NAME') ? (string) DB_NAME : '';
        return \substr(\md5((string) $host . $salt), 0, 10);
    }

    private static function get_blog_id(): int {
        return \function_exists('get_current_blog_id') ? (int) \get_current_blog_id() : 0;
    }

    private static function scope_suffix(): string {
        $sapi = \php_sapi_name();
        $sapiTag = $sapi ? \substr(\preg_replace('/[^a-z0-9]/i', '', strtolower($sapi)), 0, 6) : 'web';
        $suffix = self::get_hostname_key() . '_' . self::get_blog_id() . '_' . $sapiTag;
        $override = \apply_filters('dfehc_baseline_scope_suffix', null, $suffix);
        if (\is_string($override) && $override !== '') {
            return $override;
        }
        return $suffix;
    }

    private static function get_cache_key(string $suffix): string {
        return self::LOAD_CACHE_TRANSIENT . '_' . $suffix;
    }

    private static function get_spike_key(string $suffix): string {
        return self::LOAD_SPIKE_TRANSIENT . '_' . $suffix;
    }

    private static function get_baseline_value(string $name) {
        return \is_multisite() ? \get_site_transient($name) : \get_transient($name);
    }

    private static function set_baseline_value(string $name, $value, int $exp): void {
        if (\is_multisite()) {
            self::set_site_transient_noautoload($name, $value, $exp);
        } else {
            self::set_transient_noautoload($name, $value, $exp);
        }
    }

    private static function delete_baseline_value(string $name): void {
        if (\is_multisite()) {
            \delete_site_transient($name);
        } else {
            \delete_transient($name);
        }
    }

    private static function get_scoped_transient(string $key) {
        return \is_multisite() ? \get_site_transient($key) : \get_transient($key);
    }

    private static function set_scoped_transient_noautoload(string $key, $value, int $ttl): void {
        if (\is_multisite()) {
            self::set_site_transient_noautoload($key, $value, $ttl);
        } else {
            self::set_transient_noautoload($key, $value, $ttl);
        }
    }

    private static function delete_scoped_transient(string $key): void {
        if (\is_multisite()) {
            \delete_site_transient($key);
        } else {
            \delete_transient($key);
        }
    }

    private static function set_transient_noautoload(string $key, $value, int $ttl): void {
        $jitter = (\function_exists('random_int') ? \random_int(0, 5) : 0);
        $ttl = \max(1, $ttl + $jitter);
        if (\function_exists('wp_using_ext_object_cache') && \wp_using_ext_object_cache()) {
            \wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $ttl);
            return;
        }
        \set_transient($key, $value, $ttl);
        global $wpdb;
        if (!isset($wpdb->options)) {
            return;
        }
        $opt_key = "_transient_$key";
        $opt_key_to = "_transient_timeout_$key";
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
        if ($autoload === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $wpdb->suppress_errors(false);
    }

    private static function set_site_transient_noautoload(string $key, $value, int $ttl): void {
        $jitter = (\function_exists('random_int') ? \random_int(0, 5) : 0);
        $ttl = \max(1, $ttl + $jitter);
        if (\function_exists('wp_using_ext_object_cache') && \wp_using_ext_object_cache()) {
            \wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $ttl);
            return;
        }
        \set_site_transient($key, $value, $ttl);
    }
}

\add_action('init', [Dfehc_ServerLoadEstimator::class, 'maybe_calibrate_during_cron']);
\add_action('template_redirect', [Dfehc_ServerLoadEstimator::class, 'maybe_calibrate_if_idle']);

\add_filter('heartbeat_settings', function ($settings) {
    if (!\class_exists(Dfehc_ServerLoadEstimator::class)) {
        return $settings;
    }
    $load = Dfehc_ServerLoadEstimator::get_server_load();
    if ($load === false) {
        return $settings;
    }
    $ths = \wp_parse_args(\apply_filters('dfehc_heartbeat_thresholds', []), [
        'low'    => 20,
        'medium' => 50,
        'high'   => 75,
    ]);
    $suggested = null;
    if (!\is_admin() && !\current_user_can('edit_posts')) {
        $suggested = $load < $ths['low'] ? 50 : ($load < $ths['medium'] ? 60 : ($load < $ths['high'] ? 120 : 180));
    } elseif (\current_user_can('manage_options')) {
        $suggested = $load < $ths['high'] ? 20 : 40;
    } elseif (\current_user_can('edit_others_posts')) {
        $suggested = $load < $ths['high'] ? 30 : 60;
    }
    if ($suggested !== null) {
        $suggested = (int) \min(\max($suggested, 15), 300);
        if (isset($settings['interval']) && \is_numeric($settings['interval'])) {
            $current = (int) $settings['interval'];
            $settings['interval'] = (int) \min(\max(\max($current, $suggested), 15), 300);
        } else {
            $settings['interval'] = $suggested;
        }
    }
    return $settings;
}, 5);