<?php
namespace DynamicHeartbeat;

defined('ABSPATH') or die();

require_once plugin_dir_path(__FILE__) . 'unclogger-db.php';
require_once plugin_dir_path(__FILE__) . 'db-health.php';

class DfehcUnclogger
{
    protected $db;
    protected $config;

    protected $default_settings = [
        'auto_cleanup'        => false,
        'post_revision_limit' => 20,
    ];

    protected array $allowed_tools = [
        'delete_trashed_posts',
        'delete_revisions',
        'delete_auto_drafts',
        'delete_orphaned_postmeta',
        'delete_expired_transients',
        'count_expired_transients',
        'convert_to_innodb',
        'optimize_tables',
        'optimize_all',
        'clear_woocommerce_cache',
    ];

    public function __construct()
    {
        if (defined('WP_CLI') && WP_CLI) {
            $base = rtrim(self::get_plugin_path(), '/\\');
            $cli  = $base . '/cli-helper.php';
            if (!is_file($cli)) {
                $cli = $base . '/defibrillator/cli-helper.php';
            }
            if (is_file($cli)) {
                require_once $cli;
            }
        }

        $this->db = new DfehcUncloggerDb();
        $this->set_default_settings();

        load_plugin_textdomain(
            'dynamic-front-end-heartbeat-control',
            false,
            dirname(plugin_basename(__FILE__)) . '/languages'
        );

        add_action('rest_api_init', [$this, 'register_rest_routes']);
    }

    public static function get_plugin_path()
    {
        return plugin_dir_path(__FILE__);
    }

    public function set_default_settings()
    {
        if (!get_option('dfehc_unclogger_settings')) {
            update_option('dfehc_unclogger_settings', $this->default_settings, true);
        }
    }

    public function get_settings()
    {
        $settings = get_option('dfehc_unclogger_settings', $this->default_settings);
        return [
            'success' => true,
            'data' => $settings,
        ];
    }

    public function set_setting($req)
    {
        $data = $req->get_json_params();
        if (!is_array($data)) {
            return new \WP_Error('invalid_body', 'Invalid JSON payload.');
        }

        $setting = sanitize_text_field($data['setting'] ?? '');
        $value   = $data['value'] ?? null;

        $known = ['auto_cleanup', 'post_revision_limit'];
        if (!in_array($setting, $known, true)) {
            return new \WP_Error('invalid_setting', 'Unknown setting key.');
        }

        if ($setting === 'auto_cleanup') {
            $value = (bool) $value;
        } elseif ($setting === 'post_revision_limit') {
            $value = max(0, (int) $value);
        }

        $settings = get_option('dfehc_unclogger_settings', $this->default_settings);
        $settings[$setting] = $value;
        update_option('dfehc_unclogger_settings', $settings, true);

        if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('[DFEHC] Setting updated: ' . $setting . ' => ' . wp_json_encode($value));
        }

        return [
            'success' => true,
            'data' => $settings,
        ];
    }

    protected function get_max_load(string $context = 'interactive'): float
    {
        $interactive = (float) apply_filters('dfehc_optimize_max_load', 45.0);
        $background  = (float) apply_filters('dfehc_optimize_max_load_background', 5.0);
        return $context === 'background' ? $background : $interactive;
    }

    public function optimize_db($req)
    {
        $tool = sanitize_key($req->get_param('tool'));

        $uid = get_current_user_id();
        $ip  = isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
        $rl_key   = 'dfehc_optimize_rl_' . ($uid ? 'u' . $uid : 'ip' . md5($ip));
        $rl_limit = (int) apply_filters('dfehc_optimize_rl_limit', 10);
        $rl_win   = (int) apply_filters('dfehc_optimize_rl_window', 60);
        $rl_cnt   = (int) get_transient($rl_key);
        if ($rl_cnt >= $rl_limit) {
            return new \WP_Error('rate_limited', 'Too many requests. Please try again shortly.', ['status' => 429]);
        }
        set_transient($rl_key, $rl_cnt + 1, $rl_win);

        if (get_transient('dfehc_optimizing')) {
            return new \WP_Error('already_running', 'Optimization is already in progress.');
        }

        if (!$tool || !in_array($tool, $this->allowed_tools, true) || !method_exists($this->db, $tool)) {
            return new \WP_Error('invalid_tool', 'No valid optimization tool specified.');
        }

        $load = function_exists('dfehc_get_server_load') ? (float) dfehc_get_server_load() : 0.0;
        if ($load > $this->get_max_load('interactive')) {
            return new \WP_Error('server_busy', 'Server load is too high to run optimization safely.');
        }

        set_transient('dfehc_optimizing', 1, 300);

        if ($tool === 'optimize_all') {
            wp_schedule_single_event(time() + 10, 'dfehc_async_optimize_all');
            delete_transient('dfehc_optimizing');
            return [
                'success' => true,
                'data' => ['scheduled' => true, 'tool' => $tool],
            ];
        }

        $result = null;
        try {
            $result = call_user_func([$this, 'guarded_run_tool'], $tool);
            if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                error_log('[DFEHC] Ran tool: ' . $tool . ' -> ' . wp_json_encode($result));
            }
        } catch (\Throwable $e) {
            delete_transient('dfehc_optimizing');
            return new \WP_Error('optimize_failed', 'Optimization failed: ' . $e->getMessage());
        }

        delete_transient('dfehc_optimizing');

        return [
            'success' => true,
            'data' => [
                'result'   => $result,
                'tool'     => $tool,
                'settings' => get_option('dfehc_unclogger_settings', $this->default_settings),
            ],
        ];
    }

    protected function guarded_run_tool(string $tool)
    {
        if (!method_exists($this->db, $tool)) {
            throw new \RuntimeException('Unknown tool: ' . $tool);
        }
        if (in_array($tool, ['convert_to_innodb', 'optimize_tables', 'optimize_all'], true)) {
            $load = function_exists('dfehc_get_server_load') ? (float) dfehc_get_server_load() : 0.0;
            if ($load > $this->get_max_load('interactive')) {
                throw new \RuntimeException('Server load spiked during operation.');
            }
        }
        return call_user_func([$this->db, $tool]);
    }

    public function register_rest_routes()
    {
        register_rest_route('dfehc-unclogger/v1', '/get/', [
            'methods'             => \WP_REST_Server::READABLE,
            'permission_callback' => __NAMESPACE__ . '\\dfehc_permission_check',
            'callback'            => [$this, 'get_settings'],
        ]);

        register_rest_route('dfehc-unclogger/v1', '/optimize-db/(?P<tool>[^/]+)', [
            'methods'             => \WP_REST_Server::CREATABLE,
            'permission_callback' => __NAMESPACE__ . '\\dfehc_permission_check',
            'callback'            => [$this, 'optimize_db'],
            'args'                => [
                'tool' => [
                    'required' => true,
                    'sanitize_callback' => 'sanitize_key',
                ],
            ],
        ]);

        register_rest_route('dfehc-unclogger/v1', '/set/', [
            'methods'             => \WP_REST_Server::CREATABLE,
            'permission_callback' => __NAMESPACE__ . '\\dfehc_permission_check',
            'callback'            => [$this, 'set_setting'],
        ]);
    }

    public function __call($method, $args)
    {
        if (method_exists($this->db, $method)) {
            return call_user_func_array([$this->db, $method], $args);
        }
        throw new \BadMethodCallException("Method {$method} does not exist.");
    }
}

function dfehc_permission_check()
{
    return (bool) apply_filters('dfehc_unclogger_permission_check', current_user_can('manage_options'));
}

function dfehc_async_optimize_all()
{
    if (get_transient('dfehc_optimizing')) {
        return;
    }

    $load = function_exists('dfehc_get_server_load') ? (float) dfehc_get_server_load() : 0.0;
    $max  = (float) apply_filters('dfehc_optimize_max_load_background', 5.0);
    if ($load > $max) {
        return;
    }

    set_transient('dfehc_optimizing', 1, 300);

    $prev = ignore_user_abort(true);
    if (function_exists('set_time_limit')) {
        @set_time_limit(60);
    }

    try {
        $db = class_exists(__NAMESPACE__ . '\\DfehcUncloggerDb') ? new DfehcUncloggerDb() : new \DynamicHeartbeat\DfehcUncloggerDb();
        $db->optimize_all();
        if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('[DFEHC] optimize_all completed.');
        }
    } finally {
        ignore_user_abort($prev);
        delete_transient('dfehc_optimizing');
    }
}

add_action('dfehc_async_optimize_all', __NAMESPACE__ . '\\dfehc_async_optimize_all');

if (defined('WP_CLI') && WP_CLI) {
    \WP_CLI::add_command('dfehc optimize', function () {
        if (get_transient('dfehc_optimizing')) {
            \WP_CLI::error('Optimization already in progress.');
        }
        $load = function_exists('dfehc_get_server_load') ? (float) dfehc_get_server_load() : 0.0;
        $max  = (float) apply_filters('dfehc_optimize_max_load_background', 5.0);
        if ($load > $max) {
            \WP_CLI::error('Server load too high to proceed.');
        }
        set_transient('dfehc_optimizing', 1, 300);
        $prev = ignore_user_abort(true);
        if (function_exists('set_time_limit')) {
            @set_time_limit(60);
        }
        try {
            $db = class_exists(__NAMESPACE__ . '\\DfehcUncloggerDb') ? new DfehcUncloggerDb() : new \DynamicHeartbeat\DfehcUncloggerDb();
            $db->optimize_all();
            \WP_CLI::success('Database optimized successfully.');
        } finally {
            ignore_user_abort($prev);
            delete_transient('dfehc_optimizing');
        }
    });
}

$dfehc_unclogger = new \DynamicHeartbeat\DfehcUnclogger();

if (!class_exists(__NAMESPACE__ . '\\DfehcUncloggerDb')) {
class DfehcUncloggerDb
{
    private function sanitize_ident(string $name): string
    {
        return preg_replace('/[^A-Za-z0-9$_]/', '', $name);
    }

    public function get_database_size()
    {
        global $wpdb;
        $sql = "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1)
                FROM information_schema.tables
                WHERE table_schema = %s
                GROUP BY table_schema";
        return (float) $wpdb->get_var($wpdb->prepare($sql, $wpdb->dbname));
    }

    public function count_trashed_posts()
    {
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, 'trash'));
    }

    public function delete_trashed_posts()
    {
        global $wpdb;
        $sql = "DELETE FROM {$wpdb->posts} WHERE post_status = %s";
        return (int) $wpdb->query($wpdb->prepare($sql, 'trash'));
    }

    public function count_revisions()
    {
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, 'revision'));
    }

    public function delete_revisions()
    {
        global $wpdb;
        $sql = "DELETE FROM {$wpdb->posts} WHERE post_type = %s";
        return (int) $wpdb->query($wpdb->prepare($sql, 'revision'));
    }

    public function count_auto_drafts()
    {
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, 'auto-draft'));
    }

    public function delete_auto_drafts()
    {
        global $wpdb;
        $sql = "DELETE FROM {$wpdb->posts} WHERE post_status = %s";
        return (int) $wpdb->query($wpdb->prepare($sql, 'auto-draft'));
    }

    public function count_orphaned_postmeta()
    {
        global $wpdb;
        $sql = "SELECT COUNT(*)
                FROM {$wpdb->postmeta} pm
                LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id
                WHERE p.ID IS NULL";
        return (int) $wpdb->get_var($sql);
    }

    public function delete_orphaned_postmeta()
    {
        global $wpdb;
        $sql = "DELETE pm
                FROM {$wpdb->postmeta} pm
                LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id
                WHERE p.ID IS NULL";
        return (int) $wpdb->query($sql);
    }

    public function count_woocommerce_transients()
    {
        global $wpdb;
        $like_val = $wpdb->esc_like('_transient_woocommerce_') . '%';
        $like_to  = $wpdb->esc_like('_transient_timeout_woocommerce_') . '%';
        $sql = "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, $like_val, $like_to));
    }

    public function delete_woocommerce_transients()
    {
        global $wpdb;
        $like_val = $wpdb->esc_like('_transient_woocommerce_') . '%';
        $like_to  = $wpdb->esc_like('_transient_timeout_woocommerce_') . '%';
        $sql1 = $wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $like_val);
        $sql2 = $wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $like_to);
        $c1 = (int) $wpdb->query($sql1);
        $c2 = (int) $wpdb->query($sql2);
        return $c1 + $c2;
    }

    public function clear_woocommerce_cache()
{
    if (class_exists('\WC_Cache_Helper')) {
        try { \WC_Cache_Helper::get_transient_version('product', true); } catch (\Throwable $e) {}
        try { \WC_Cache_Helper::get_transient_version('shipping', true); } catch (\Throwable $e) {}
        try { \WC_Cache_Helper::get_transient_version('orders', true); } catch (\Throwable $e) {}
    }
    if (function_exists('\wc_delete_product_transients')) {
        \wc_delete_product_transients();
    }
    return true;
}


    public function count_expired_transients()
    {
        global $wpdb;
        $like_to = $wpdb->esc_like('_transient_timeout_') . '%';
        $sql = "SELECT COUNT(*)
                FROM {$wpdb->options}
                WHERE option_name LIKE %s
                  AND CAST(option_value AS UNSIGNED) < %d";
        return (int) $wpdb->get_var($wpdb->prepare($sql, $like_to, time()));
    }

    public function delete_expired_transients()
    {
        global $wpdb;
        $like_to = $wpdb->esc_like('_transient_timeout_') . '%';
        $names_sql = "SELECT REPLACE(option_name, %s, '') AS tname
                      FROM {$wpdb->options}
                      WHERE option_name LIKE %s
                        AND CAST(option_value AS UNSIGNED) < %d
                      LIMIT %d";
        $batch = (int) apply_filters('dfehc_delete_transients_batch', 1000);
        $time_budget = (float) apply_filters('dfehc_delete_transients_time_budget', 2.5);
        $start = microtime(true);

        $count = 0;
        do {
            $rows = $wpdb->get_col($wpdb->prepare($names_sql, '_transient_timeout_', $like_to, time(), $batch));
            if (!$rows) {
                break;
            }
            foreach ($rows as $name) {
                $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name = %s", "_transient_{$name}"));
                $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name = %s", "_transient_timeout_{$name}"));
                $count++;
                if ((microtime(true) - $start) > $time_budget) {
                    break 2;
                }
            }
        } while (true);

        return $count;
    }

    public function count_tables_with_different_prefix()
    {
        global $wpdb;
        $sql = "SELECT COUNT(TABLE_NAME)
                FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s
                  AND TABLE_NAME NOT LIKE %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, $wpdb->dbname, $wpdb->base_prefix . '%'));
    }

    public function list_tables_with_different_prefix()
    {
        global $wpdb;
        $sql = "SELECT TABLE_NAME
                FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s
                  AND TABLE_NAME NOT LIKE %s";
        $results = $wpdb->get_col($wpdb->prepare($sql, $wpdb->dbname, $wpdb->base_prefix . '%'));
        return implode(', ', array_map('esc_html', (array) $results));
    }

    public function drop_tables_with_different_prefix()
    {
        global $wpdb;
        $allowed_prefixes = (array) apply_filters('dfehc_unclogger_allowed_drop_prefixes', []);
        if (empty($allowed_prefixes)) {
            return 0;
        }
        $sql = "SELECT TABLE_NAME
                FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s
                  AND (" . implode(' OR ', array_fill(0, count($allowed_prefixes), "TABLE_NAME LIKE %s")) . ")";
        $params = [$wpdb->dbname];
        foreach ($allowed_prefixes as $p) {
            $params[] = $wpdb->esc_like($p) . '%';
        }
        $prepared = $wpdb->prepare($sql, $params);
        $results  = $wpdb->get_col($prepared);

        $count = 0;
        foreach ((array) $results as $table) {
            $safe = $this->sanitize_ident($table);
            if ($safe === '') {
                continue;
            }
            $wpdb->query("DROP TABLE IF EXISTS `{$safe}`");
            $count++;
        }
        return $count;
    }

    public function count_myisam_tables()
    {
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s AND ENGINE = 'MyISAM'";
        return (int) $wpdb->get_var($wpdb->prepare($sql, $wpdb->dbname));
    }

    public function list_myisam_tables()
    {
        global $wpdb;
        $sql = "SELECT TABLE_NAME FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s AND ENGINE = 'MyISAM'";
        $results = $wpdb->get_col($wpdb->prepare($sql, $wpdb->dbname));
        return implode(', ', array_map('esc_html', (array) $results));
    }

    public function convert_to_innodb()
    {
        global $wpdb;
        $sql = "SELECT TABLE_NAME FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s AND ENGINE = 'MyISAM' AND TABLE_NAME LIKE %s";
        $tables = $wpdb->get_col($wpdb->prepare($sql, $wpdb->dbname, $wpdb->base_prefix . '%'));

        $count = 0;
        foreach ((array) $tables as $t) {
            $safe = $this->sanitize_ident($t);
            if ($safe === '') {
                continue;
            }
            $wpdb->query("ALTER TABLE `{$safe}` ENGINE=InnoDB");
            $count++;
        }
        return $count;
    }

    public function optimize_tables()
    {
        global $wpdb;
        $sql = "SELECT TABLE_NAME FROM information_schema.TABLES
                WHERE TABLE_SCHEMA = %s AND TABLE_NAME LIKE %s";
        $tables = $wpdb->get_col($wpdb->prepare($sql, $wpdb->dbname, $wpdb->base_prefix . '%'));

        $count = 0;
        foreach ((array) $tables as $t) {
            $safe = $this->sanitize_ident($t);
            if ($safe === '') {
                continue;
            }
            $wpdb->query("OPTIMIZE TABLE `{$safe}`");
            $count++;
        }
        return $count;
    }

    public function count_tables()
    {
        global $wpdb;
        $sql = "SELECT COUNT(TABLE_NAME) FROM information_schema.TABLES WHERE TABLE_SCHEMA = %s";
        return (int) $wpdb->get_var($wpdb->prepare($sql, $wpdb->dbname));
    }

    public function optimize_all()
    {
        $this->delete_trashed_posts();
        $this->delete_revisions();
        $this->delete_auto_drafts();
        $this->delete_orphaned_postmeta();
        $this->delete_expired_transients();
        $this->convert_to_innodb();
        $this->optimize_tables();
    }

    public function set_wp_post_revisions($value)
    {
        if (!isset($this->config)) {
            return new \WP_Error('config_missing', 'Config instance not set.');
        }
        if ($value === 'default') {
            return $this->config->remove('constant', 'WP_POST_REVISIONS');
        }
        return $this->config->update('constant', 'WP_POST_REVISIONS', $value);
    }
}
}