<?php
declare(strict_types=1);

if (!defined('DFEHC_OPTIONS_PREFIX')) define('DFEHC_OPTIONS_PREFIX', 'dfehc_');
if (!defined('DFEHC_OPTION_MIN_INTERVAL')) define('DFEHC_OPTION_MIN_INTERVAL', DFEHC_OPTIONS_PREFIX . 'min_interval');
if (!defined('DFEHC_OPTION_MAX_INTERVAL')) define('DFEHC_OPTION_MAX_INTERVAL', DFEHC_OPTIONS_PREFIX . 'max_interval');
if (!defined('DFEHC_OPTION_PRIORITY_SLIDER')) define('DFEHC_OPTION_PRIORITY_SLIDER', DFEHC_OPTIONS_PREFIX . 'priority_slider');
if (!defined('DFEHC_OPTION_EMA_ALPHA')) define('DFEHC_OPTION_EMA_ALPHA', DFEHC_OPTIONS_PREFIX . 'ema_alpha');
if (!defined('DFEHC_OPTION_MAX_SERVER_LOAD')) define('DFEHC_OPTION_MAX_SERVER_LOAD', DFEHC_OPTIONS_PREFIX . 'max_server_load');
if (!defined('DFEHC_OPTION_MAX_RESPONSE_TIME')) define('DFEHC_OPTION_MAX_RESPONSE_TIME', DFEHC_OPTIONS_PREFIX . 'max_response_time');
if (!defined('DFEHC_OPTION_SMA_WINDOW')) define('DFEHC_OPTION_SMA_WINDOW', DFEHC_OPTIONS_PREFIX . 'sma_window');
if (!defined('DFEHC_OPTION_MAX_DECREASE_RATE')) define('DFEHC_OPTION_MAX_DECREASE_RATE', DFEHC_OPTIONS_PREFIX . 'max_decrease_rate');

if (!defined('DFEHC_DEFAULT_MIN_INTERVAL')) define('DFEHC_DEFAULT_MIN_INTERVAL', 15);
if (!defined('DFEHC_DEFAULT_MAX_INTERVAL')) define('DFEHC_DEFAULT_MAX_INTERVAL', 300);
if (!defined('DFEHC_DEFAULT_MAX_SERVER_LOAD')) define('DFEHC_DEFAULT_MAX_SERVER_LOAD', 85);
if (!defined('DFEHC_DEFAULT_MAX_RESPONSE_TIME')) define('DFEHC_DEFAULT_MAX_RESPONSE_TIME', 5.0);
if (!defined('DFEHC_DEFAULT_EMA_ALPHA')) define('DFEHC_DEFAULT_EMA_ALPHA', 0.4);
if (!defined('DFEHC_DEFAULT_SMA_WINDOW')) define('DFEHC_DEFAULT_SMA_WINDOW', 5);
if (!defined('DFEHC_DEFAULT_MAX_DECREASE_RATE')) define('DFEHC_DEFAULT_MAX_DECREASE_RATE', 0.25);
if (!defined('DFEHC_DEFAULT_EMA_TTL')) define('DFEHC_DEFAULT_EMA_TTL', 600);

if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}

if (!function_exists('dfehc_scoped_key')) {
    function dfehc_scoped_key(string $base): string {
        return "{$base}_" . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}

if (!function_exists('dfehc_store_lockfree')) {
    function dfehc_store_lockfree(string $key, $value, int $ttl): bool {
        $group = defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc';
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            wp_cache_set($key, $value, $group, $ttl);
            return true;
        }
        return set_transient($key, $value, $ttl);
    }
}

if (!function_exists('dfehc_set_transient_noautoload')) {
    function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void {
        $group = defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc';
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            if (function_exists('wp_cache_add')) {
                if (!wp_cache_add($key, $value, $group, $expiration)) {
                    wp_cache_set($key, $value, $group, $expiration);
                }
            } else {
                wp_cache_set($key, $value, $group, $expiration);
            }
            return;
        }
        set_transient($key, $value, $expiration);
        global $wpdb;
        if (!isset($wpdb->options)) return;
        $opt_key = "_transient_$key";
        $opt_key_to = "_transient_timeout_$key";
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
        if ($autoload === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $wpdb->suppress_errors(false);
    }
}

if (!function_exists('dfehc_set_transient')) {
    function dfehc_set_transient(string $key, float $value, float $interval): void {
        $ttl = (int) apply_filters('dfehc_transient_ttl', max(60, (int) ceil($interval) * 2), $key, $value, $interval);
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($key, $value, $ttl);
    }
}

if (!function_exists('dfehc_clamp')) {
    function dfehc_clamp(float $v, float $lo, float $hi): float {
        return max($lo, min($hi, $v));
    }
}

if (!function_exists('dfehc_weighted_sum')) {
    function dfehc_weighted_sum(array $factors, array $weights): float {
        $sum = 0.0;
        foreach ($factors as $k => $v) {
            $sum += ((float) ($weights[$k] ?? 0.0)) * (float) $v;
        }
        return $sum;
    }
}

if (!function_exists('dfehc_normalize_weights')) {
    function dfehc_normalize_weights(array $weights): array {
        $total = array_sum($weights);
        if ($total <= 0) {
            $n = max(1, count($weights));
            $equal = 1 / $n;
            foreach ($weights as $k => $_) {
                $weights[$k] = $equal;
            }
            return $weights;
        }
        foreach ($weights as $k => $w) {
            $weights[$k] = $w / $total;
        }
        return $weights;
    }
}

if (!function_exists('dfehc_apply_exponential_moving_average')) {
    function dfehc_apply_exponential_moving_average(float $current): float {
        $alpha = dfehc_clamp((float) get_option(DFEHC_OPTION_EMA_ALPHA, DFEHC_DEFAULT_EMA_ALPHA), 0.01, 1.0);
        $key   = dfehc_scoped_key('dfehc_ema');
        $prev  = get_transient($key);
        $ema   = ($prev === false) ? $current : $alpha * $current + (1 - $alpha) * (float) $prev;
        $ttl_default = max(DFEHC_DEFAULT_EMA_TTL, (int) get_option(DFEHC_OPTION_MAX_INTERVAL, DFEHC_DEFAULT_MAX_INTERVAL) * 2);
        $ttl_default = (int) dfehc_clamp($ttl_default, 60, 86400);
        $ttl = (int) apply_filters('dfehc_ema_ttl', $ttl_default, $current, $ema);
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($key, $ema, $ttl);
        return $ema;
    }
}

if (!function_exists('dfehc_calculate_recommended_interval')) {
    function dfehc_calculate_recommended_interval(float $time_elapsed, float $load_average, float $server_response_time): float {
        $min_interval      = max(1, (int) get_option(DFEHC_OPTION_MIN_INTERVAL, DFEHC_DEFAULT_MIN_INTERVAL));
        $max_interval      = max($min_interval, (int) get_option(DFEHC_OPTION_MAX_INTERVAL, DFEHC_DEFAULT_MAX_INTERVAL));
        $max_server_load   = max(0.1, (float) get_option(DFEHC_OPTION_MAX_SERVER_LOAD, DFEHC_DEFAULT_MAX_SERVER_LOAD));
        $max_response_time = max(0.1, (float) get_option(DFEHC_OPTION_MAX_RESPONSE_TIME, DFEHC_DEFAULT_MAX_RESPONSE_TIME));

        $custom_norm = apply_filters('dfehc_normalize_load', null, $load_average);
        if (is_numeric($custom_norm)) {
            $la = dfehc_clamp((float) $custom_norm, 0.0, 1.0);
        } else {
            if ($load_average <= 1.0) {
                $la = dfehc_clamp($load_average, 0.0, 1.0);
            } elseif ($load_average <= 100.0) {
                $la = dfehc_clamp($load_average / 100.0, 0.0, 1.0);
            } else {
                $assumed_cores = (float) apply_filters('dfehc_assumed_cores_for_normalization', 8.0);
                $la = dfehc_clamp($load_average / max(1.0, $assumed_cores), 0.0, 1.0);
            }
        }

        $msl_ratio = $max_server_load > 1.0 ? ($max_server_load / 100.0) : $max_server_load;
        if ($msl_ratio <= 0) $msl_ratio = 1.0;
        $server_load_factor = dfehc_clamp($la / $msl_ratio, 0.0, 1.0);

        $rt_units = apply_filters('dfehc_response_time_is_ms', null, $server_response_time);
        $rt = (float) $server_response_time;
        if ($rt_units === true) {
            $rt = $rt / 1000.0;
        } elseif ($rt_units === null) {
            if ($rt > ($max_response_time * 3) && $rt <= 60000.0) {
                $rt = $rt / 1000.0;
            }
        }
        $rt = max(0.0, $rt);
        $response_time_factor = $rt > 0 ? dfehc_clamp($rt / $max_response_time, 0.0, 1.0) : 0.0;

        $factors = [
            'user_activity' => dfehc_clamp($time_elapsed / $max_interval, 0.0, 1.0),
            'server_load'   => $server_load_factor,
            'response_time' => $response_time_factor,
        ];
        $factors = (array) apply_filters('dfehc_interval_factors', $factors, $time_elapsed, $load_average, $server_response_time);

        $slider  = dfehc_clamp((float) get_option(DFEHC_OPTION_PRIORITY_SLIDER, 0.0), -1.0, 1.0);
        $weights = [
            'user_activity' => 0.4 - 0.2 * $slider,
            'server_load'   => (0.6 + 0.2 * $slider) / 2,
            'response_time' => (0.6 + 0.2 * $slider) / 2,
        ];
        $weights = (array) apply_filters('dfehc_interval_weights', $weights, $slider);
        $weights = dfehc_normalize_weights($weights);

        $raw      = $min_interval + dfehc_weighted_sum($factors, $weights) * ($max_interval - $min_interval);
        $smoothed = dfehc_apply_exponential_moving_average($raw);
        $lagged   = dfehc_defensive_stance($smoothed);

        $final = (float) apply_filters('dfehc_interval_snap', $lagged, $min_interval, $max_interval);
        $final = dfehc_clamp($final, (float) $min_interval, (float) $max_interval);

        return $final;
    }
}

if (!function_exists('dfehc_calculate_interval_based_on_duration')) {
    function dfehc_calculate_interval_based_on_duration(float $avg_duration, float $load_average): float {
        $min_interval = max(1, (int) get_option(DFEHC_OPTION_MIN_INTERVAL, DFEHC_DEFAULT_MIN_INTERVAL));
        $max_interval = max($min_interval, (int) get_option(DFEHC_OPTION_MAX_INTERVAL, DFEHC_DEFAULT_MAX_INTERVAL));
        if ($avg_duration <= $min_interval) return (float) $min_interval;
        if ($avg_duration >= $max_interval) return (float) $max_interval;
        $proposed = dfehc_calculate_recommended_interval($avg_duration, $load_average, 0.0);
        return dfehc_defensive_stance($proposed);
    }
}

if (!function_exists('dfehc_smooth_moving')) {
    function dfehc_smooth_moving(array $values): float {
        if (!$values) return 0.0;
        $window = max(1, (int) get_option(DFEHC_OPTION_SMA_WINDOW, DFEHC_DEFAULT_SMA_WINDOW));
        $subset = array_slice($values, -$window);
        if (!$subset) return 0.0;
        return array_sum($subset) / count($subset);
    }
}

if (!function_exists('dfehc_defensive_stance')) {
    function dfehc_defensive_stance(float $proposed): float {
        $key      = dfehc_scoped_key('dfehc_prev_int');
        $previous = get_transient($key);
        if ($previous === false) {
            $ttl = (int) apply_filters('dfehc_prev_interval_ttl', 1800);
            $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
            dfehc_set_transient_noautoload($key, $proposed, $ttl);
            return $proposed;
        }
        $previous  = (float) $previous;
        $max_drop  = dfehc_clamp((float) get_option(DFEHC_OPTION_MAX_DECREASE_RATE, DFEHC_DEFAULT_MAX_DECREASE_RATE), 0.0, 0.95);
        $max_rise  = dfehc_clamp((float) apply_filters('dfehc_max_increase_rate', 0.5), 0.0, 5.0);
        $lower     = $previous * (1 - $max_drop);
        $upper     = $previous * (1 + $max_rise);
        $final     = dfehc_clamp($proposed, $lower, $upper);
        $ttl = (int) apply_filters('dfehc_prev_interval_ttl', 1800);
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($key, $final, $ttl);
        return $final;
    }
}