<?php
declare(strict_types=1);

if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool
    {
        return $needle === '' || strpos($haystack, $needle) === 0;
    }
}

if (!defined('DFEHC_CACHE_GROUP')) {
    define('DFEHC_CACHE_GROUP', 'dfehc');
}
if (!defined('DFEHC_SERVER_LOAD_TTL')) {
    define('DFEHC_SERVER_LOAD_TTL', 180);
}
if (!defined('DFEHC_SERVER_LOAD_CACHE_KEY')) {
    define('DFEHC_SERVER_LOAD_CACHE_KEY', 'dfehc:server_load');
}
if (!defined('DFEHC_SERVER_LOAD_PAYLOAD_KEY')) {
    define('DFEHC_SERVER_LOAD_PAYLOAD_KEY', 'dfehc_server_load_payload');
}

if (!function_exists('dfehc_server_load_ttl')) {
    function dfehc_server_load_ttl(): int
    {
        return (int) apply_filters('dfehc_server_load_ttl', (int) DFEHC_SERVER_LOAD_TTL);
    }
}

if (!function_exists('dfehc_unknown_load')) {
    function dfehc_unknown_load(): float
    {
        static $v;
        if ($v === null) {
            $v = (float) apply_filters('dfehc_unknown_load', 0.404);
        }
        return $v;
    }
}

if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string
    {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int
    {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}

if (!function_exists('dfehc_key')) {
    function dfehc_key(string $base): string
    {
        return $base . '_' . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}

if (!function_exists('dfehc_client_ip')) {
    function dfehc_client_ip(): string
    {
        $ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
        $trusted = (array) apply_filters('dfehc_trusted_proxies', []);
        if ($trusted && in_array($ip, $trusted, true)) {
            $headers = (array) apply_filters('dfehc_proxy_ip_headers', ['HTTP_X_FORWARDED_FOR', 'HTTP_CF_CONNECTING_IP', 'HTTP_X_REAL_IP']);
            foreach ($headers as $h) {
                if (!empty($_SERVER[$h])) {
                    $raw = (string) $_SERVER[$h];
                    $parts = array_map('trim', explode(',', $raw));
                    $cand = end($parts);
                    if ($cand) {
                        return (string) apply_filters('dfehc_client_ip', $cand);
                    }
                }
            }
        }
        return (string) apply_filters('dfehc_client_ip', $ip);
    }
}

if (!function_exists('dfehc_get_redis_server')) {
    function dfehc_get_redis_server(): string
    {
        $h = getenv('REDIS_HOST');
        return $h ? (string) $h : '127.0.0.1';
    }
}
if (!function_exists('dfehc_get_redis_port')) {
    function dfehc_get_redis_port(): int
    {
        $p = getenv('REDIS_PORT');
        return $p && ctype_digit((string) $p) ? (int) $p : 6379;
    }
}
if (!function_exists('dfehc_get_memcached_server')) {
    function dfehc_get_memcached_server(): string
    {
        $h = getenv('MEMCACHED_HOST');
        return $h ? (string) $h : '127.0.0.1';
    }
}
if (!function_exists('dfehc_get_memcached_port')) {
    function dfehc_get_memcached_port(): int
    {
        $p = getenv('MEMCACHED_PORT');
        return $p && ctype_digit((string) $p) ? (int) $p : 11211;
    }
}

function _dfehc_get_cache_client(): array
{
    static $cached = null;
    static $ts = 0;
    $retryAfter = (int) apply_filters('dfehc_cache_retry_after', 60);
    if ($cached !== null && ($cached['type'] !== 'none' || $ts > time() - $retryAfter)) {
        return $cached;
    }
    $ts = time();
    global $wp_object_cache;
    if (is_object($wp_object_cache) && isset($wp_object_cache->redis) && $wp_object_cache->redis instanceof Redis) {
        return $cached = ['client' => $wp_object_cache->redis, 'type' => 'redis'];
    }
    if (is_object($wp_object_cache) && isset($wp_object_cache->mc) && $wp_object_cache->mc instanceof Memcached) {
        return $cached = ['client' => $wp_object_cache->mc, 'type' => 'memcached'];
    }
    if (class_exists('Redis')) {
        try {
            $redis = new Redis();
            if ($redis->connect(dfehc_get_redis_server(), dfehc_get_redis_port(), 1.0)) {
                $pass = apply_filters('dfehc_redis_auth', getenv('REDIS_PASSWORD') ?: null);
                $user = apply_filters('dfehc_redis_user', getenv('REDIS_USERNAME') ?: null);
                if ($user && $pass && method_exists($redis, 'auth')) {
                    $redis->auth([$user, $pass]);
                } elseif ($pass && method_exists($redis, 'auth')) {
                    $redis->auth($pass);
                }
                return $cached = ['client' => $redis, 'type' => 'redis'];
            }
        } catch (Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('DFEHC Redis connect error: ' . $e->getMessage());
            }
        }
    }
    if (class_exists('Memcached')) {
        try {
            $mc = new Memcached('dfehc');
            if (!$mc->getServerList()) {
                $mc->addServer(dfehc_get_memcached_server(), dfehc_get_memcached_port());
            }
            $user = getenv('MEMCACHED_USERNAME');
            $pass = getenv('MEMCACHED_PASSWORD');
            if ($user && $pass && method_exists($mc, 'setSaslAuthData')) {
                $mc->setOption(Memcached::OPT_BINARY_PROTOCOL, true);
                $mc->setSaslAuthData($user, $pass);
            }
            $versions = $mc->getVersion();
            $first = is_array($versions) ? reset($versions) : false;
            $ok = $first && $first !== '0.0.0';
            if ($ok) {
                return $cached = ['client' => $mc, 'type' => 'memcached'];
            }
        } catch (Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('DFEHC Memcached connect error: ' . $e->getMessage());
            }
        }
    }
    return $cached = ['client' => null, 'type' => 'none'];
}

function dfehc_cache_server_load(float $value): void
{
    ['client' => $client, 'type' => $type] = _dfehc_get_cache_client();
    if (!$client) {
        return;
    }
    $key = dfehc_key(DFEHC_SERVER_LOAD_CACHE_KEY);
    $ttl = dfehc_server_load_ttl();
    try {
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        if ($type === 'redis') {
            $client->setex($key, $ttl, $value);
        } elseif ($type === 'memcached') {
            $client->set($key, $value, $ttl);
        }
    } catch (Throwable $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('DFEHC cache write error: ' . $e->getMessage());
        }
    }
}

function dfehc_get_server_load(): float
{
    $payloadKey = dfehc_key(DFEHC_SERVER_LOAD_PAYLOAD_KEY);
    $payload = null;
    if (wp_using_ext_object_cache()) {
        $payload = wp_cache_get($payloadKey, DFEHC_CACHE_GROUP);
        if ($payload === false) {
            $payload = null;
        }
    } else {
        $payload = get_transient($payloadKey);
    }
    if (!(is_array($payload) && isset($payload['raw'], $payload['cores'], $payload['source']))) {
        if (!dfehc_load_acquire_lock()) {
            return dfehc_unknown_load();
        }
        try {
            $data = dfehc_detect_load_raw_with_source();
            $payload = [
                'raw' => (float) $data['load'],
                'cores' => dfehc_get_cpu_cores(),
                'source' => (string) $data['source'],
            ];
            $ttl = dfehc_server_load_ttl();
            $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
            if (wp_using_ext_object_cache()) {
                wp_cache_set($payloadKey, $payload, DFEHC_CACHE_GROUP, $ttl);
            } else {
                set_transient($payloadKey, $payload, $ttl);
            }
        } finally {
            dfehc_load_release_lock();
        }
    }
    $raw = (float) $payload['raw'];
    $cores = (int) ($payload['cores'] ?: dfehc_get_cpu_cores());
    $source = (string) $payload['source'];
    $divide = (bool) apply_filters('dfehc_divide_cpu_load', true, $raw, $cores, $source);
    $load = ($source === 'cpu_load' && $divide && $cores > 0) ? $raw / $cores : $raw;
    $load = max(0.0, (float) $load);
    return (float) apply_filters('dfehc_contextual_load_value', $load, $source);
}

function dfehc_detect_load_raw_with_source(): array
{
    if (function_exists('sys_getloadavg')) {
        $arr = sys_getloadavg();
        if ($arr && isset($arr[0]) && $arr[0] >= 0) {
            return ['load' => (float) $arr[0], 'source' => 'cpu_load'];
        }
    }
    if (is_readable('/proc/loadavg')) {
        $txt = file_get_contents('/proc/loadavg');
        if ($txt !== false) {
            $parts = explode(' ', trim($txt));
            if (isset($parts[0])) {
                return ['load' => (float) $parts[0], 'source' => 'cpu_load'];
            }
        }
    }
    $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
    if (function_exists('shell_exec') && !in_array('shell_exec', $disabled, true) && !ini_get('open_basedir')) {
        $out = shell_exec('LANG=C uptime 2>&1');
        if ($out && preg_match('/load average[s]?:\s*([0-9.]+)/', $out, $m)) {
            return ['load' => (float) $m[1], 'source' => 'cpu_load'];
        }
    }
    if (defined('DFEHC_PLUGIN_PATH')) {
        $est = rtrim((string) DFEHC_PLUGIN_PATH, "/\\") . '/defibrillator/load-estimator.php';
        if (file_exists($est)) {
            require_once $est;
            if (class_exists('DynamicHeartbeat\\Dfehc_ServerLoadEstimator')) {
                $pct = DynamicHeartbeat\Dfehc_ServerLoadEstimator::get_server_load();
                if (is_numeric($pct)) {
                    $cores = dfehc_get_cpu_cores();
                    $cpuLoad = ((float) $pct / 100.0) * max(1, $cores);
                    return ['load' => $cpuLoad, 'source' => 'cpu_load'];
                }
            }
        }
    }
    if (function_exists('do_action')) {
        do_action('dfehc_load_detection_fell_back');
    }
    return ['load' => dfehc_unknown_load(), 'source' => 'fallback'];
}

function dfehc_get_cpu_cores(): int
{
    static $cores = null;
    if ($cores !== null) {
        return (int) $cores;
    }
    $override = getenv('DFEHC_CPU_CORES');
    if ($override && ctype_digit((string) $override) && (int) $override > 0) {
        $cores = (int) $override;
        return (int) $cores;
    }
    if (is_readable('/sys/fs/cgroup/cpu.max')) {
        $line = file_get_contents('/sys/fs/cgroup/cpu.max');
        if ($line !== false) {
            [$quota, $period] = explode(' ', trim($line));
            if ($quota !== 'max') {
                $q = (int) $quota;
                $p = (int) $period;
                if ($q > 0 && $p > 0) {
                    $cores = max(1, (int) ceil($q / $p));
                    $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
                    return (int) $cores;
                }
            }
        }
    }
    if (is_readable('/proc/self/cgroup')) {
        $content = file_get_contents('/proc/self/cgroup');
        if ($content !== false && preg_match('/^[0-9]+:[^:]*cpu[^:]*:(.+)$/m', $content, $m)) {
            $path = '/' . ltrim(trim($m[1]), '/');
            $base = '/sys/fs/cgroup' . $path;
            $quotaFile = "$base/cpu.cfs_quota_us";
            $periodFile = "$base/cpu.cfs_period_us";
            if (is_readable($quotaFile) && is_readable($periodFile)) {
                $quota = (int) file_get_contents($quotaFile);
                $period = (int) file_get_contents($periodFile);
                if ($quota > 0 && $period > 0) {
                    $cores = max(1, (int) ceil($quota / $period));
                    $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
                    return (int) $cores;
                }
            }
        }
    }
    if (is_readable('/sys/fs/cgroup/cpu/cpu.cfs_quota_us') && is_readable('/sys/fs/cgroup/cpu/cpu.cfs_period_us')) {
        $quota = (int) file_get_contents('/sys/fs/cgroup/cpu/cpu.cfs_quota_us');
        $period = (int) file_get_contents('/sys/fs/cgroup/cpu/cpu.cfs_period_us');
        if ($quota > 0 && $period > 0) {
            $cores = max(1, (int) ceil($quota / $period));
            $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
            return (int) $cores;
        }
    }
    $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
    if (function_exists('shell_exec') && !in_array('shell_exec', $disabled, true) && !ini_get('open_basedir')) {
        $n = shell_exec('nproc 2>/dev/null');
        if ($n && ctype_digit(trim((string) $n))) {
            $cores = max(1, (int) trim((string) $n));
            $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
            return (int) $cores;
        }
    }
    if (is_readable('/proc/cpuinfo')) {
        $info = file_get_contents('/proc/cpuinfo');
        if ($info !== false) {
            $cnt = preg_match_all('/^processor/m', $info);
            if ($cnt) {
                $cores = (int) $cnt;
                $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
                return (int) $cores;
            }
        }
    }
    $cores = 1;
    $cores = (int) apply_filters('dfehc_cpu_cores', (int) $cores);
    return (int) $cores;
}

function dfehc_log_server_load(): void
{
    $load = dfehc_get_server_load();
    $optKey = 'dfehc_server_load_logs_' . dfehc_blog_id() . '_' . dfehc_host_token();
    $logs = get_option($optKey, []);
    if (!is_array($logs)) {
        $logs = [];
    }
    $now = time();
    $cutoff = $now - DAY_IN_SECONDS;
    $logs = array_filter(
        $logs,
        static function (array $row) use ($cutoff): bool {
            return isset($row['timestamp']) && $row['timestamp'] >= $cutoff;
        }
    );
    if (count($logs) > 2000) {
        $logs = array_slice($logs, -2000);
    }
    $logs[] = ['timestamp' => $now, 'load' => $load];
    update_option($optKey, array_values($logs), false);
}
add_action('dfehc_log_server_load_hook', 'dfehc_log_server_load');

function dfehc_get_server_load_ajax_handler(): void
{
    $allow_public = apply_filters('dfehc_allow_public_server_load', false);
    if (!$allow_public) {
        $action = 'get_server_load';
        $nonce_action = 'dfehc-' . $action;
        $valid = function_exists('check_ajax_referer')
            ? check_ajax_referer($nonce_action, 'nonce', false)
            : wp_verify_nonce((string)($_POST['nonce'] ?? $_GET['nonce'] ?? ''), $nonce_action);
        if (!$valid) {
            wp_send_json_error(['message' => 'Invalid nonce.'], 403);
        }
        $cap = apply_filters('dfehc_required_capability', 'read');
        if (!current_user_can($cap)) {
            wp_send_json_error(['message' => 'Not authorised.'], 403);
        }
    } else {
        $ip = dfehc_client_ip();
        $rk = dfehc_key('dfehc_rl_' . md5($ip));
        $cnt = (int) get_transient($rk);
        $limit = (int) apply_filters('dfehc_public_rate_limit', 60);
        $win   = (int) apply_filters('dfehc_public_rate_window', 60);
        if ($cnt >= $limit) {
            wp_send_json_error(['message' => 'rate_limited'], 429);
        }
        set_transient($rk, $cnt + 1, $win);
    }
    nocache_headers();
    wp_send_json_success(dfehc_get_server_load_persistent());
}
add_action('wp_ajax_get_server_load', 'dfehc_get_server_load_ajax_handler');

add_action('init', static function (): void {
    if (apply_filters('dfehc_allow_public_server_load', false)) {
        add_action('wp_ajax_nopriv_get_server_load', 'dfehc_get_server_load_ajax_handler');
    }
}, 0);

function dfehc_get_server_load_persistent(): float
{
    static $cached = null;
    if ($cached !== null) {
        return (float) $cached;
    }
    ['client' => $client] = _dfehc_get_cache_client();
    $val = false;
    $key = dfehc_key(DFEHC_SERVER_LOAD_CACHE_KEY);
    if ($client) {
        try {
            $val = $client->get($key);
        } catch (Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('DFEHC cache read error: ' . $e->getMessage());
            }
        }
    }
    if ($val !== false && $val !== '') {
        $cached = max(0.0, (float) $val);
        return (float) $cached;
    }
    $fresh = dfehc_get_server_load();
    $fresh = max(0.0, (float) $fresh);
    dfehc_cache_server_load($fresh);
    if (wp_using_ext_object_cache()) {
        $ttl = dfehc_server_load_ttl();
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        wp_cache_set($key, $fresh, DFEHC_CACHE_GROUP, $ttl);
    }
    $cached = $fresh;
    return (float) $cached;
}

add_filter('cron_schedules', static function (array $schedules): array {
    if (!isset($schedules['dfehc_minute'])) {
        $schedules['dfehc_minute'] = [
            'interval' => 60,
            'display' => __('Server load (DFEHC)', 'dfehc'),
        ];
    }
    return $schedules;
}, 5);

$__dfehc_schedule = static function (): void {
    if (!apply_filters('dfehc_enable_load_logging', true)) {
        return;
    }
    if (!wp_next_scheduled('dfehc_log_server_load_hook')) {
        try {
            $now = time();
            $start = $now - ($now % 60) + 60;
            wp_schedule_event($start, 'dfehc_minute', 'dfehc_log_server_load_hook');
        } catch (Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('DFEHC scheduling error: ' . $e->getMessage());
            }
        }
    }
};

register_activation_hook(__FILE__, $__dfehc_schedule);

add_action('init', static function () use ($__dfehc_schedule): void {
    $__dfehc_schedule();
}, 1);

function dfehc_load_acquire_lock(): bool
{
    $key = dfehc_key('dfehc_load_lock');
    if (class_exists('WP_Lock')) {
        $lock = new WP_Lock($key, 30);
        if ($lock->acquire()) {
            $GLOBALS['dfehc_load_lock'] = $lock;
            return true;
        }
        return false;
    }
    if (function_exists('wp_cache_add') && wp_cache_add($key, 1, DFEHC_CACHE_GROUP, 30)) {
        $GLOBALS['dfehc_load_lock_cache_key'] = $key;
        return true;
    }
    if (false !== get_transient($key)) {
        return false;
    }
    if (set_transient($key, 1, 30)) {
        $GLOBALS['dfehc_load_lock_transient_key'] = $key;
        return true;
    }
    return false;
}

function dfehc_load_release_lock(): void
{
    if (isset($GLOBALS['dfehc_load_lock']) && $GLOBALS['dfehc_load_lock'] instanceof WP_Lock) {
        $GLOBALS['dfehc_load_lock']->release();
        unset($GLOBALS['dfehc_load_lock']);
        return;
    }
    if (isset($GLOBALS['dfehc_load_lock_cache_key'])) {
        wp_cache_delete($GLOBALS['dfehc_load_lock_cache_key'], DFEHC_CACHE_GROUP);
        unset($GLOBALS['dfehc_load_lock_cache_key']);
        return;
    }
    if (isset($GLOBALS['dfehc_load_lock_transient_key'])) {
        delete_transient($GLOBALS['dfehc_load_lock_transient_key']);
        unset($GLOBALS['dfehc_load_lock_transient_key']);
    }
}