<?php
declare(strict_types=1);

defined('DFEHC_DEFAULT_RESPONSE_TIME') || define('DFEHC_DEFAULT_RESPONSE_TIME', 50.0);
defined('DFEHC_HEAD_NEG_TTL') || define('DFEHC_HEAD_NEG_TTL', 600);
defined('DFEHC_HEAD_POS_TTL') || define('DFEHC_HEAD_POS_TTL', WEEK_IN_SECONDS);
defined('DFEHC_SPIKE_OPT_EPS') || define('DFEHC_SPIKE_OPT_EPS', 0.1);
defined('DFEHC_BASELINE_EXP') || define('DFEHC_BASELINE_EXP', 7 * DAY_IN_SECONDS);
defined('DFEHC_CACHE_GROUP') || define('DFEHC_CACHE_GROUP', 'dfehc');

if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}

if (!function_exists('dfehc_key')) {
    function dfehc_key(string $base): string {
        return $base . '_' . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}

if (!function_exists('dfehc_store_lockfree')) {
    function dfehc_store_lockfree(string $key, $value, int $ttl): bool {
        if (function_exists('wp_cache_add') && wp_cache_add($key, $value, DFEHC_CACHE_GROUP, $ttl)) {
            return true;
        }
        return set_transient($key, $value, $ttl);
    }
}

if (!function_exists('dfehc_set_transient_noautoload')) {
    function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void {
        if (wp_using_ext_object_cache()) {
            if (function_exists('wp_cache_add')) {
                if (!wp_cache_add($key, $value, DFEHC_CACHE_GROUP, $expiration)) {
                    wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
                }
            } else {
                wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
            }
            return;
        }
        set_transient($key, $value, $expiration);
        global $wpdb;
        $opt_key = "_transient_$key";
        $opt_key_to = "_transient_timeout_$key";
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
        if ($autoload === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $wpdb->suppress_errors(false);
    }
}

if (!function_exists('dfehc_client_ip')) {
    function dfehc_client_ip(): string {
        $ip = (string)($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
        $trusted = (array) apply_filters('dfehc_trusted_proxies', []);
        if ($trusted && in_array($ip, $trusted, true)) {
            $headers = (array) apply_filters('dfehc_proxy_ip_headers', ['HTTP_X_FORWARDED_FOR', 'HTTP_CF_CONNECTING_IP', 'HTTP_X_REAL_IP']);
            foreach ($headers as $h) {
                if (!empty($_SERVER[$h])) {
                    $raw = (string) $_SERVER[$h];
                    $parts = array_map('trim', explode(',', $raw));
                    $cand = end($parts);
                    if ($cand) {
                        return (string) apply_filters('dfehc_client_ip', $cand);
                    }
                }
            }
        }
        return (string) apply_filters('dfehc_client_ip', $ip);
    }
}

function dfehc_get_server_response_time(): array
{
    $now = time();
    $default_ms = (float) apply_filters('dfehc_default_response_time', DFEHC_DEFAULT_RESPONSE_TIME);

    $defaults = [
        'main_response_ms' => null,
        'db_response_ms' => null,
        'method' => '',
        'measurements' => [],
        'recalibrated' => false,
        'timestamp' => current_time('mysql'),
        'baseline_used' => null,
        'spike_score' => 0.0,
        'ts_unix' => $now,
    ];

    $cacheKey = dfehc_key('dfehc_cached_response_data');
    $cached = get_transient($cacheKey);
    if ($cached !== false && is_array($cached)) {
        return array_merge($defaults, $cached);
    }

    if (function_exists('dfehc_is_high_traffic') && dfehc_is_high_traffic()) {
        $high = [
            'main_response_ms' => $default_ms,
            'db_response_ms' => null,
            'method' => 'throttled',
            'measurements' => [],
            'recalibrated' => false,
            'timestamp' => current_time('mysql'),
            'baseline_used' => null,
            'spike_score' => 0.0,
            'ts_unix' => $now,
        ];
        $ttl = (int) apply_filters('dfehc_high_traffic_cache_expiration', 300);
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($cacheKey, $high, $ttl);
        return $high;
    }

    if (!dfehc_acquire_lock()) {
        return array_merge($defaults, is_array($cached) ? $cached : []);
    }

    try {
        $results = dfehc_perform_response_measurements($default_ms);
        $baselineKey = dfehc_key('dfehc_baseline_response_data');
        $spikeKey = dfehc_key('dfehc_spike_score');
        $baseline = get_transient($baselineKey);
        $spike = (float) get_transient($spikeKey);
        $max_age = (int) apply_filters('dfehc_max_baseline_age', DFEHC_BASELINE_EXP);

        if (is_array($baseline)) {
            $ts = isset($baseline['ts_unix']) && is_numeric($baseline['ts_unix']) ? (int) $baseline['ts_unix'] : strtotime($baseline['timestamp'] ?? 'now');
            if ($now - (int) $ts > $max_age) {
                delete_transient($baselineKey);
                $baseline = false;
            }
        }

        if ($baseline === false && $results['method'] === 'http_loopback' && $results['main_response_ms'] !== null && count((array) $results['measurements']) >= (int) apply_filters('dfehc_baseline_min_samples', 2)) {
            $exp = (int) apply_filters('dfehc_baseline_expiration', DFEHC_BASELINE_EXP);
            $results['timestamp'] = current_time('mysql');
            $results['ts_unix'] = $now;
            $exp += function_exists('random_int') ? random_int(0, 5) : 0;
            dfehc_set_transient_noautoload($baselineKey, $results, $exp);
            $baseline = $results;
            $spike = 0.0;
        }

        $results['baseline_used'] = $baseline;

        if (is_array($baseline) && $results['method'] === 'http_loopback' && isset($results['main_response_ms'], $baseline['main_response_ms'])) {
            $base_ms = max(1.0, (float) $baseline['main_response_ms']);
            $curr_ms = (float) $results['main_response_ms'];
            $factor = (float) apply_filters('dfehc_spike_threshold_factor', 2.0);
            $raw_inc = max(0.0, $curr_ms / $base_ms - $factor);
            $floor = (float) apply_filters('dfehc_spike_increment_floor', 0.25);
            $cap = (float) apply_filters('dfehc_spike_increment_cap', 3.0);
            $increment = min($cap, max($floor, $raw_inc));
            $decay = (float) apply_filters('dfehc_spike_decay', 0.25);
            $threshold = (float) apply_filters('dfehc_recalibrate_threshold', 5.0);

            if ($curr_ms > $base_ms * $factor) {
                $spike += $increment;
            } else {
                $spike = max(0.0, $spike - $decay);
            }

            if ($spike >= $threshold) {
                $results['timestamp'] = current_time('mysql');
                $results['ts_unix'] = $now;
                $exp = (int) apply_filters('dfehc_baseline_expiration', DFEHC_BASELINE_EXP);
                $exp += function_exists('random_int') ? random_int(0, 5) : 0;
                dfehc_set_transient_noautoload($baselineKey, $results, $exp);
                $spike = 0.0;
                $results['recalibrated'] = true;
            }
        }

        $results['spike_score'] = $spike;
        $prev_spike = (float) get_transient($spikeKey);

        if (abs($spike - $prev_spike) >= DFEHC_SPIKE_OPT_EPS) {
            $ttl = DFEHC_BASELINE_EXP + (function_exists('random_int') ? random_int(0, 5) : 0);
            dfehc_set_transient_noautoload($spikeKey, $spike, $ttl);
        }

        $exp = (int) apply_filters('dfehc_cache_expiration', 3 * MINUTE_IN_SECONDS);
        $exp += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($cacheKey, $results, $exp);

        return array_merge($defaults, $results);
    } finally {
        dfehc_release_lock();
    }
}

function dfehc_perform_response_measurements(float $default_ms): array
{
    $now = time();
    $r = [
        'main_response_ms' => null,
        'db_response_ms' => null,
        'method' => 'http_loopback',
        'measurements' => [],
        'recalibrated' => false,
        'timestamp' => current_time('mysql'),
        'ts_unix' => $now,
    ];

    if (apply_filters('dfehc_disable_loopback', false) || (function_exists('wp_is_recovery_mode') && wp_is_recovery_mode())) {
        $r['method'] = 'throttled';
        $r['main_response_ms'] = $default_ms;
        $r['db_response_ms'] = $default_ms;
        return $r;
    }

    global $wpdb;
    try {
        $db_start = microtime(true);
        $wpdb->query('SELECT 1');
        $r['db_response_ms'] = (microtime(true) - $db_start) * 1000;
    } catch (\Throwable $e) {
        $r['db_response_ms'] = $default_ms;
    }

    $rest = function_exists('get_rest_url') ? get_rest_url() : '';
    $url = $rest ?: (function_exists('home_url') ? home_url('/wp-json/') : '/wp-json/');
    if (function_exists('get_option') && !get_option('permalink_structure')) {
        $url = add_query_arg('rest_route', '/', home_url('/index.php'));
    }

    $ajax_fallback = function_exists('admin_url') ? admin_url('admin-ajax.php?action=dfehc_ping') : '/wp-admin/admin-ajax.php?action=dfehc_ping';
    $use_ajax_fallback = false;

    $home_host = function_exists('home_url') ? (string) parse_url(home_url(), PHP_URL_HOST) : '';
    $headers = (array) apply_filters('dfehc_probe_headers', [
        'Host' => $home_host ?: '',
        'Cache-Control' => 'max-age=0, must-revalidate',
        'X-DFEHC-Probe' => '1',
    ]);
    $hard_deadline = microtime(true) + (float) apply_filters('dfehc_total_timeout', (int) apply_filters('dfehc_request_timeout', 10) + 2);

    if (defined('WP_HTTP_BLOCK_EXTERNAL') && WP_HTTP_BLOCK_EXTERNAL) {
        $probe_host = parse_url($url, PHP_URL_HOST);
        $accessible = getenv('WP_ACCESSIBLE_HOSTS') ?: (defined('WP_ACCESSIBLE_HOSTS') ? WP_ACCESSIBLE_HOSTS : '');
        $allowed_hosts = array_filter(array_map('trim', explode(',', (string) $accessible)));
        $is_same_host = $home_host && $probe_host && strcasecmp((string) $home_host, (string) $probe_host) === 0;
        if (!$is_same_host && !in_array((string) $probe_host, $allowed_hosts, true)) {
            $use_ajax_fallback = true;
        }
    }

    if ($use_ajax_fallback || empty($url)) {
        $url = $ajax_fallback;
        $r['method'] = 'ajax_loopback';
    }

    $n = max(1, min((int) apply_filters('dfehc_num_requests', 3), 5));
    $sleep_us = (int) apply_filters('dfehc_request_pause_us', 50000);
    $timeout = (int) apply_filters('dfehc_request_timeout', 10);
    $sslverify = (bool) apply_filters('dfehc_ssl_verify', true);
    $get_fallback = (bool) apply_filters('dfehc_use_get_fallback', true);
    $use_head = (bool) apply_filters('dfehc_use_head_method', true);
    $redirection = (int) apply_filters('dfehc_redirection', 1);

    $scheme = (string) parse_url($url, PHP_URL_SCHEME);
    $head_key = dfehc_key('dfehc_head_supported_' . md5($scheme . '|' . $url));
    $head_supported = get_transient($head_key);
    if ($head_supported === false) {
        $head_supported = null;
    }

    $negKey = dfehc_key('dfehc_probe_fail');
    if (get_transient($negKey)) {
        $r['method'] = 'failed';
        $r['main_response_ms'] = $default_ms;
        if ($r['db_response_ms'] === null) {
            $r['db_response_ms'] = $default_ms;
        }
        return $r;
    }

    $times = [];

    for ($i = 0; $i < $n; $i++) {
        $remaining = $hard_deadline - microtime(true);
        if ($remaining <= 0) {
            break;
        }

        $probe_url = add_query_arg('_dfehc_ts', sprintf('%.6f', microtime(true)), $url);

        $args = [
            'timeout' => max(1, min((int) ceil($remaining), $timeout)),
            'sslverify' => $sslverify,
            'headers' => $headers,
            'redirection' => min($redirection, 3),
            'reject_unsafe_urls' => true,
            'blocking' => true,
            'decompress' => false,
            'limit_response_size' => (int) apply_filters('dfehc_limit_response_size', 512),
        ];

        $start = microtime(true);
        $resp = null;

        if ($use_head && $head_supported !== 0) {
            $resp = wp_remote_head($probe_url, $args);
            if (is_wp_error($resp) || wp_remote_retrieve_response_code($resp) >= 400) {
                if ($head_supported === null) {
                    $ttl = (int) apply_filters('dfehc_head_negative_ttl', DFEHC_HEAD_NEG_TTL);
                    $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
                    dfehc_set_transient_noautoload($head_key, 0, $ttl);
                }
                $resp = null;
            } else {
                if ($head_supported === null) {
                    $ttl = (int) apply_filters('dfehc_head_positive_ttl', DFEHC_HEAD_POS_TTL);
                    $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
                    dfehc_set_transient_noautoload($head_key, 1, $ttl);
                }
            }
        }

        if ($resp === null && $get_fallback) {
            $resp = wp_remote_get($probe_url, $args);
        }

        $code = is_wp_error($resp) ? 0 : wp_remote_retrieve_response_code($resp);
        if (($code >= 200 && $code < 300) || $code === 304) {
            $times[] = (microtime(true) - $start) * 1000;
        }

        if (count($times) >= $n) {
            break;
        }

        if ($i < $n - 1 && $sleep_us > 0) {
            usleep($sleep_us);
        }
    }

    if ($times) {
        sort($times, SORT_NUMERIC);
        if (count($times) >= 3 && (bool) apply_filters('dfehc_trim_extremes', true)) {
            array_shift($times);
            array_pop($times);
        }
        $cnt = count($times);
        $r['measurements'] = $times;
        $r['main_response_ms'] = $cnt % 2 ? (float) $times[intdiv($cnt, 2)] : (float) (($times[$cnt / 2 - 1] + $times[$cnt / 2]) / 2);
    } else {
        $ttl = (int) apply_filters('dfehc_probe_fail_ttl', 60);
        $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
        dfehc_set_transient_noautoload($negKey, 1, $ttl);
        $r['method'] = 'failed';
    }

    if ($r['main_response_ms'] === null) {
        $r['main_response_ms'] = $default_ms;
    }
    if ($r['db_response_ms'] === null) {
        $r['db_response_ms'] = $default_ms;
    }

    return $r;
}

function dfehc_ping_handler(): void {
    $ip = dfehc_client_ip();
    $k = dfehc_key('dfehc_ping_rl_' . md5($ip));
    $window = (int) apply_filters('dfehc_ping_rl_ttl', 2);
    $limit  = (int) apply_filters('dfehc_ping_rl_limit', 2);
    $cnt = (int) get_transient($k);
    if ($cnt >= $limit) {
        status_header(429);
        nocache_headers();
        wp_send_json_error('rate_limited', 429);
    }
    dfehc_set_transient_noautoload($k, $cnt + 1, $window);
    nocache_headers();
    wp_send_json_success('ok');
}

add_action('wp_ajax_dfehc_ping', 'dfehc_ping_handler');
if (apply_filters('dfehc_enable_public_ping', true)) {
    add_action('wp_ajax_nopriv_dfehc_ping', 'dfehc_ping_handler');
}

if (!function_exists('dfehc_acquire_lock')) {
    function dfehc_acquire_lock(): bool
    {
        $key = dfehc_key('dfehc_measure_lock');

        if (class_exists('WP_Lock')) {
            $lock = new WP_Lock($key, 60);
            if ($lock->acquire()) {
                $GLOBALS['dfehc_rt_lock'] = $lock;
                return true;
            }
            return false;
        }

        if (function_exists('wp_cache_add') && wp_cache_add($key, 1, DFEHC_CACHE_GROUP, 60)) {
            $GLOBALS['dfehc_rt_lock_cache_key'] = $key;
            return true;
        }

        if (false !== get_transient($key)) {
            return false;
        }

        if (set_transient($key, 1, 60)) {
            $GLOBALS['dfehc_rt_lock_transient_key'] = $key;
            return true;
        }

        return false;
    }
}

if (!function_exists('dfehc_release_lock')) {
    function dfehc_release_lock(): void
    {
        if (isset($GLOBALS['dfehc_rt_lock']) && $GLOBALS['dfehc_rt_lock'] instanceof WP_Lock) {
            $GLOBALS['dfehc_rt_lock']->release();
            unset($GLOBALS['dfehc_rt_lock']);
            return;
        }

        if (isset($GLOBALS['dfehc_rt_lock_cache_key'])) {
            wp_cache_delete($GLOBALS['dfehc_rt_lock_cache_key'], DFEHC_CACHE_GROUP);
            unset($GLOBALS['dfehc_rt_lock_cache_key']);
            return;
        }

        if (isset($GLOBALS['dfehc_rt_lock_transient_key'])) {
            delete_transient($GLOBALS['dfehc_rt_lock_transient_key']);
            unset($GLOBALS['dfehc_rt_lock_transient_key']);
        }
    }
}

if (!function_exists('dfehc_is_high_traffic')) {
    function dfehc_is_high_traffic(): bool
    {
        $flag_key = dfehc_key('dfehc_high_traffic_flag');
        $flag = get_transient($flag_key);
        if ($flag !== false) {
            return (bool) $flag;
        }

        $threshold = (int) apply_filters('dfehc_high_traffic_threshold', 100);
        $cnt_key = dfehc_key('dfehc_cached_visitor_cnt');
        $count = get_transient($cnt_key);
        if ($count === false) {
            $count = (int) apply_filters('dfehc_website_visitors', 0);
            dfehc_set_transient_noautoload($cnt_key, (int) $count, 60);
        }

        $load = apply_filters('dfehc_current_server_load', null);
        if (is_numeric($load)) {
            $max_load = (float) apply_filters('dfehc_high_traffic_load_threshold', 85.0);
            if ((float) $load >= $max_load) {
                dfehc_set_transient_noautoload($flag_key, 1, 60);
                return true;
            }
        }

        $high = $count >= $threshold;
        dfehc_set_transient_noautoload($flag_key, $high ? 1 : 0, 60);

        return $high;
    }
}