<?php
declare(strict_types=1);

defined('DFEHC_SERVER_LOAD_TTL') || define('DFEHC_SERVER_LOAD_TTL', 60);
defined('DFEHC_SENTINEL_NO_LOAD') || define('DFEHC_SENTINEL_NO_LOAD', 0.404);
defined('DFEHC_SYSTEM_LOAD_KEY') || define('DFEHC_SYSTEM_LOAD_KEY', 'dfehc_system_load_avg');
defined('DFEHC_CACHE_GROUP') || define('DFEHC_CACHE_GROUP', 'dfehc');

if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}

if (!function_exists('dfehc_scoped_key')) {
    function dfehc_scoped_key(string $base): string {
        return $base . '_' . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}

if (!function_exists('dfehc_set_transient_noautoload')) {
    function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void {
        $jitter = function_exists('random_int') ? random_int(0, 5) : 0;
        $expiration = max(1, $expiration + $jitter);
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
            return;
        }
        set_transient($key, $value, $expiration);
        if (isset($GLOBALS['wpdb'])) {
            global $wpdb;
            if (!isset($wpdb->options)) {
                return;
            }
            $wpdb->suppress_errors(true);
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->options} SET autoload='no' WHERE option_name=%s AND autoload='yes' LIMIT 1", "_transient_$key"));
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->options} SET autoload='no' WHERE option_name=%s AND autoload='yes' LIMIT 1", "_transient_timeout_$key"));
            $wpdb->suppress_errors(false);
        }
    }
}

if (!function_exists('dfehc_exec_with_timeout')) {
    function dfehc_exec_with_timeout(string $cmd, float $timeoutSec = 1.0): string {
        $timeoutSec = max(0.1, min(5.0, $timeoutSec));
        $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
        if (ini_get('open_basedir')) {
            return '';
        }
        $can_proc = function_exists('proc_open') && !in_array('proc_open', $disabled, true);
        if ($can_proc) {
            $descriptorspec = [['pipe','r'],['pipe','w'],['pipe','w']];
            $process = @proc_open($cmd, $descriptorspec, $pipes);
            if (!is_resource($process)) return '';
            foreach ($pipes as $p) { @stream_set_blocking($p, false); @stream_set_timeout($p, (int)ceil($timeoutSec)); }
            $out = ''; $err = '';
            $start = microtime(true);
            $spins = 0;
            while (true) {
                $out .= @stream_get_contents($pipes[1]) ?: '';
                $err .= @stream_get_contents($pipes[2]) ?: '';
                $status = @proc_get_status($process);
                if (!$status || !$status['running']) break;
                if ((microtime(true) - $start) > $timeoutSec) { @proc_terminate($process); break; }
                $spins++;
                usleep($spins > 10 ? 25000 : 10000);
            }
            foreach ($pipes as $p) { @fclose($p); }
            @proc_close($process);
            return trim($out);
        }
        $can_shell = function_exists('shell_exec') && !in_array('shell_exec', $disabled, true);
        if ($can_shell) {
            return trim((string) @shell_exec($cmd));
        }
        return '';
    }
}

if (!function_exists('dfehc_get_cpu_cores')) {
    function dfehc_get_cpu_cores(): int {
        static $cached = null;
        if ($cached !== null) return $cached;
        $tkey = dfehc_scoped_key('dfehc_cpu_cores');
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            $oc = wp_cache_get($tkey, DFEHC_CACHE_GROUP);
            if ($oc !== false && (int) $oc > 0) {
                return $cached = (int) $oc;
            }
        }
        $tc = get_transient($tkey);
        if ($tc !== false && (int) $tc > 0) {
            return $cached = (int) $tc;
        }
        $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
        if (PHP_OS_FAMILY !== 'Windows' && function_exists('shell_exec') && !in_array('shell_exec', $disabled, true) && !ini_get('open_basedir')) {
            $val = trim((string) @shell_exec('getconf _NPROCESSORS_ONLN 2>/dev/null'));
            if (ctype_digit($val) && (int) $val > 0) {
                $cores = (int) $val;
                dfehc_set_transient_noautoload($tkey, $cores, (int) apply_filters('dfehc_cpu_cores_ttl', DAY_IN_SECONDS));
                return $cached = $cores;
            }
        }
        if (PHP_OS_FAMILY === 'Windows' && !ini_get('open_basedir')) {
            $csv = dfehc_exec_with_timeout('typeperf -sc 1 "\Processor(_Total)\% Processor Time" 2>NUL', 1.5);
            if ($csv !== '') {
                $wval = dfehc_exec_with_timeout('wmic cpu get NumberOfLogicalProcessors /value 2>NUL', 1.0);
                if ($wval && preg_match('/NumberOfLogicalProcessors=(\d+)/i', $wval, $m) && (int) $m[1] > 0) {
                    $cores = (int) $m[1];
                    dfehc_set_transient_noautoload($tkey, $cores, (int) apply_filters('dfehc_cpu_cores_ttl', DAY_IN_SECONDS));
                    return $cached = $cores;
                }
            }
            $wout = dfehc_exec_with_timeout('wmic cpu get NumberOfLogicalProcessors /value 2>NUL', 1.0);
            if ($wout && preg_match('/NumberOfLogicalProcessors=(\d+)/i', $wout, $m2) && (int) $m2[1] > 0) {
                $cores = (int) $m2[1];
                dfehc_set_transient_noautoload($tkey, $cores, (int) apply_filters('dfehc_cpu_cores_ttl', DAY_IN_SECONDS));
                return $cached = $cores;
            }
        }
        if (is_readable('/proc/cpuinfo')) {
            $cnt = preg_match_all('/^processor/m', (string) @file_get_contents('/proc/cpuinfo'));
            if ($cnt > 0) {
                $cores = (int) $cnt;
                dfehc_set_transient_noautoload($tkey, $cores, (int) apply_filters('dfehc_cpu_cores_ttl', DAY_IN_SECONDS));
                return $cached = $cores;
            }
        }
        dfehc_set_transient_noautoload($tkey, 1, (int) apply_filters('dfehc_cpu_cores_ttl', DAY_IN_SECONDS));
        return $cached = 1;
    }
}

if (!function_exists('dfehc_get_system_load_average')) {
    function dfehc_get_system_load_average(): float {
        $ttl = (int) apply_filters('dfehc_system_load_ttl', DFEHC_SERVER_LOAD_TTL);
        $key = dfehc_scoped_key(DFEHC_SYSTEM_LOAD_KEY);
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            $vc = wp_cache_get($key, DFEHC_CACHE_GROUP);
            if ($vc !== false && $vc !== null) {
                $ratio = (float) $vc;
                $as_percent = (bool) apply_filters('dfehc_system_load_return_percent', false, $ratio);
                return $as_percent ? (float) round($ratio * 100, 2) : $ratio;
            }
        }
        $cache = get_transient($key);
        if ($cache !== false) {
            $ratio = (float) $cache;
            $as_percent = (bool) apply_filters('dfehc_system_load_return_percent', false, $ratio);
            return $as_percent ? (float) round($ratio * 100, 2) : $ratio;
        }

        $raw = null;
        $source = '';
        $normalized_ratio = false;

        if (function_exists('dfehc_get_server_load')) {
            $val = dfehc_get_server_load();
            if ($val !== DFEHC_SENTINEL_NO_LOAD) {
                dfehc_set_transient_noautoload($key, (float) $val, $ttl);
                $ratio = (float) $val;
                $as_percent = (bool) apply_filters('dfehc_system_load_return_percent', false, $ratio);
                return $as_percent ? (float) round($ratio * 100, 2) : $ratio;
            }
        }

        if (function_exists('sys_getloadavg')) {
            $arr = @sys_getloadavg();
            if ($arr && isset($arr[0])) {
                $raw = (float) $arr[0];
                $source = 'sys_getloadavg';
            }
        }

        if ($raw === null && PHP_OS_FAMILY !== 'Windows' && is_readable('/proc/loadavg')) {
            $parts = explode(' ', (string) @file_get_contents('/proc/loadavg'));
            if (isset($parts[0])) {
                $raw = (float) $parts[0];
                $source = 'proc_loadavg';
            }
        }

        if ($raw === null && PHP_OS_FAMILY !== 'Windows') {
            $out = dfehc_exec_with_timeout('LANG=C uptime 2>/dev/null', 1.0);
            if ($out && preg_match('/load average[s]?:\s*([0-9.]+)/', $out, $m)) {
                $raw = (float) $m[1];
                $source = 'uptime';
            }
        }

        if ($raw === null && PHP_OS_FAMILY === 'Windows' && !ini_get('open_basedir')) {
            $csv = dfehc_exec_with_timeout('typeperf -sc 1 "\Processor(_Total)\% Processor Time" 2>NUL', 1.5);
            if ($csv) {
                $lines = array_values(array_filter(array_map('trim', explode("\n", $csv))));
                $last = end($lines);
                if ($last && preg_match('/,"?([0-9.]+)"?$/', $last, $m)) {
                    $pct = (float) $m[1];
                    $raw = ($pct / 100.0) * dfehc_get_cpu_cores();
                    $source = 'typeperf';
                }
            }
        }

        if ($raw === null && PHP_OS_FAMILY === 'Windows' && !ini_get('open_basedir')) {
            $psCmd = "powershell -NoProfile -NonInteractive -Command \"\\\$v=(Get-Counter '\\Processor(_Total)\\% Processor Time').CounterSamples[0].CookedValue; [Console]::Out.WriteLine([Math]::Round(\\\$v,2))\" 2>NUL";
            $ps = dfehc_exec_with_timeout($psCmd, 1.5);
            if ($ps !== '' && is_numeric(trim($ps))) {
                $pct = (float) trim($ps);
                $raw = ($pct / 100.0) * dfehc_get_cpu_cores();
                $source = 'powershell';
            }
        }

        if ($raw === null && PHP_OS_FAMILY === 'Windows' && !ini_get('open_basedir')) {
            $out = dfehc_exec_with_timeout('wmic cpu get loadpercentage /value 2>NUL', 1.0);
            if ($out && preg_match('/loadpercentage=(\d+)/i', $out, $m)) {
                $pct = (float) $m[1];
                $raw = ($pct / 100.0) * dfehc_get_cpu_cores();
                $source = 'wmic';
            }
        }

        if ($raw === null && class_exists('\\DynamicHeartbeat\\Dfehc_ServerLoadEstimator')) {
            $est = \DynamicHeartbeat\Dfehc_ServerLoadEstimator::get_server_load();
            if (is_numeric($est)) {
                $pct = (float) $est;
                if ($pct >= 0.0 && $pct <= 100.0) {
                    $raw = $pct / 100.0;
                    $normalized_ratio = true;
                    $source = 'estimator_percent';
                } else {
                    $raw = (float) $pct;
                    $source = 'estimator_raw';
                }
            }
        }

        if ($raw === null) {
            $sentinel_ttl = (int) apply_filters('dfehc_sentinel_ttl', 5);
            dfehc_set_transient_noautoload($key, DFEHC_SENTINEL_NO_LOAD, $sentinel_ttl);
            $ratio = (float) DFEHC_SENTINEL_NO_LOAD;
            $as_percent = (bool) apply_filters('dfehc_system_load_return_percent', false, $ratio);
            return $as_percent ? (float) round($ratio * 100, 2) : $ratio;
        }

        if ($normalized_ratio) {
            $ratio = (float) $raw;
        } else {
            $divide = (bool) apply_filters('dfehc_divide_load_by_cores', true, $raw, $source);
            if ($divide) {
                $cores = dfehc_get_cpu_cores();
                $ratio = $cores > 0 ? ((float) $raw) / $cores : (float) $raw;
            } else {
                $ratio = (float) $raw;
            }
        }

        dfehc_set_transient_noautoload($key, $ratio, $ttl);
        $as_percent = (bool) apply_filters('dfehc_system_load_return_percent', false, $ratio);
        return $as_percent ? (float) round($ratio * 100, 2) : (float) $ratio;
    }
}