<?php
declare(strict_types=1);

define('DFEHC_LOAD_AVERAGES', 'dfehc_load_averages');
define('DFEHC_SERVER_LOAD', 'dfehc_server_load');
define('DFEHC_RECOMMENDED_INTERVAL', 'dfehc_recommended_interval');
define('DFEHC_CAPABILITY', 'read');
define('DFEHC_LOAD_LOCK_BASE', 'dfehc_compute_load_lock');
define('DFEHC_CACHE_GROUP', 'dfehc');

function dfehc_max_server_load(): int { static $v; if ($v === null) { $v = (int) apply_filters('dfehc_max_server_load', 85); } return $v; }
function dfehc_min_interval(): int { static $v; if ($v === null) { $v = (int) apply_filters('dfehc_min_interval', 15); } return $v; }
function dfehc_max_interval(): int { static $v; if ($v === null) { $v = (int) apply_filters('dfehc_max_interval', 300); } return $v; }
function dfehc_fallback_interval(): int { static $v; if ($v === null) { $v = (int) apply_filters('dfehc_fallback_interval', 60); } return $v; }
function dfehc_server_load_ttl(): int { static $v; if ($v === null) { $v = (int) apply_filters('dfehc_server_load_ttl', 180); } return $v; }

function dfehc_host_token(): string
{
    $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
    $salt = defined('DB_NAME') ? (string) DB_NAME : '';
    return substr(md5((string) $host . $salt), 0, 10);
}

function dfehc_scoped_key(string $base): string
{
    $blog = function_exists('get_current_blog_id') ? (string) get_current_blog_id() : '0';
    return "{$base}_{$blog}_" . dfehc_host_token();
}

function dfehc_store_lockfree(string $key, $value, int $ttl): bool
{
    if (function_exists('wp_cache_add') && wp_cache_add($key, $value, DFEHC_CACHE_GROUP, $ttl)) {
        return true;
    }
    return set_transient($key, $value, $ttl);
}

function dfehc_client_ip(): string
{
    $ip = isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
    $trusted = (array) apply_filters('dfehc_trusted_proxies', []);
    if ($trusted && in_array($ip, $trusted, true)) {
        $headers = (array) apply_filters('dfehc_proxy_ip_headers', ['HTTP_X_FORWARDED_FOR', 'HTTP_CF_CONNECTING_IP', 'HTTP_X_REAL_IP']);
        foreach ($headers as $h) {
            if (!empty($_SERVER[$h])) {
                $raw = (string) $_SERVER[$h];
                $parts = array_map('trim', explode(',', $raw));
                $cand = end($parts);
                if ($cand) {
                    return $cand;
                }
            }
        }
    }
    return $ip;
}

function dfehc_register_ajax(string $action, callable $callback): void
{
    add_action("wp_ajax_$action", $callback);
    $allow_public = false;
    if ($action === 'get_server_load') {
        $allow_public = (bool) apply_filters('dfehc_allow_public_server_load', false);
    } elseif ($action === 'dfehc_async_heartbeat') {
        $allow_public = (bool) apply_filters('dfehc_allow_public_async', false);
    } else {
        $allow_public = (bool) apply_filters("{$action}_allow_public", false);
    }
    if ($allow_public) {
        add_action("wp_ajax_nopriv_$action", $callback);
    }
}

function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void
{
    if (wp_using_ext_object_cache()) {
        if (function_exists('wp_cache_add')) {
            if (!wp_cache_add($key, $value, DFEHC_CACHE_GROUP, $expiration)) {
                wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
            }
        } else {
            wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
        }
        return;
    }
    dfehc_store_lockfree($key, $value, $expiration);
    global $wpdb;
    $opt_key = "_transient_$key";
    $opt_key_to = "_transient_timeout_$key";
    $wpdb->suppress_errors(true);
    $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
    if ($autoload === 'yes') {
        $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
    }
    $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
    if ($autoload_to === 'yes') {
        $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
    }
    $wpdb->suppress_errors(false);
}

if (!function_exists('dfehc_get_cpu_cores')) {
    function dfehc_get_cpu_cores(): int
    {
        static $cached = null;
        if ($cached !== null) {
            return (int) $cached;
        }
        $detected = 1;
        if (is_readable('/proc/cpuinfo')) {
            $cnt = preg_match_all('/^processor/m', (string) file_get_contents('/proc/cpuinfo'));
            if ($cnt > 0) {
                $detected = $cnt;
            }
        } else {
            $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
            if (function_exists('shell_exec') && !in_array('shell_exec', $disabled, true)) {
                $out = (string) (shell_exec('nproc 2>/dev/null') ?? '');
                if (ctype_digit(trim($out)) && (int) $out > 0) {
                    $detected = (int) trim($out);
                }
            }
        }
        $env_override = getenv('DFEHC_CPU_CORES');
        if ($env_override !== false && ctype_digit((string) $env_override) && (int) $env_override > 0) {
            $detected = (int) $env_override;
        }
        $detected = (int) apply_filters('dfehc_cpu_cores', $detected);
        if ($detected <= 0) {
            $detected = 1;
        }
        $cached = $detected;
        return (int) $cached;
    }
}

function dfehc_acquire_lock(string $base, int $ttl)
{
    $key = dfehc_scoped_key($base);
    if (class_exists('WP_Lock')) {
        $lock = new WP_Lock($key, $ttl);
        if ($lock->acquire()) {
            return $lock;
        }
        return null;
    }
    if (function_exists('wp_cache_add') && wp_cache_add($key, 1, DFEHC_CACHE_GROUP, $ttl)) {
        return (object) ['cache_key' => $key];
    }
    if (get_transient($key) !== false) {
        return null;
    }
    if (set_transient($key, 1, $ttl)) {
        return (object) ['transient_key' => $key];
    }
    return null;
}

function dfehc_release_lock($lock): void
{
    if ($lock instanceof WP_Lock) {
        $lock->release();
        return;
    }
    if (is_object($lock) && isset($lock->cache_key)) {
        wp_cache_delete($lock->cache_key, DFEHC_CACHE_GROUP);
        return;
    }
    if (is_object($lock) && isset($lock->transient_key)) {
        delete_transient($lock->transient_key);
    }
}

function dfehc_get_or_calculate_server_load()
{
    $key = dfehc_scoped_key(DFEHC_SERVER_LOAD);
    $load = get_transient($key);
    if ($load !== false) {
        return (float) $load;
    }
    $ttl = dfehc_server_load_ttl();
    $lock = dfehc_acquire_lock(DFEHC_LOAD_LOCK_BASE, $ttl + 5);
    if (!$lock) {
        return false;
    }
    $raw = dfehc_calculate_server_load();
    dfehc_release_lock($lock);
    if ($raw === false) {
        return false;
    }
    $cores = max(1, dfehc_get_cpu_cores());
    $load_pct = min(100.0, round($raw / $cores * 100, 2));
    dfehc_set_transient_noautoload($key, $load_pct, $ttl);
    return $load_pct;
}

function dfehc_get_server_load_ajax_handler(): void
{
    $cap = apply_filters('dfehc_required_capability', DFEHC_CAPABILITY);
    $allow_public = (bool) apply_filters('dfehc_allow_public_server_load', false);
    if (!$allow_public) {
        $nonce_action = 'dfehc-get_server_load';
        $valid = function_exists('check_ajax_referer')
            ? check_ajax_referer($nonce_action, 'nonce', false)
            : wp_verify_nonce(isset($_REQUEST['nonce']) ? (string) $_REQUEST['nonce'] : '', $nonce_action);
        if (!$valid) {
            wp_send_json_error(['message' => 'dfehc/get_server_load: invalid nonce'], 403);
        }
        if (!current_user_can($cap)) {
            wp_send_json_error(['message' => 'dfehc/get_server_load: not authorized'], 403);
        }
    } else {
        $limit = absint(apply_filters('dfehc_public_rate_limit', 30));
        $window = absint(apply_filters('dfehc_public_rate_window', 60));
        $ip = dfehc_client_ip();
        $rl_key = dfehc_scoped_key('dfehc_rl_' . md5($ip));
        $cnt = (int) get_transient($rl_key);
        if ($cnt >= $limit) {
            wp_send_json_error(['message' => 'dfehc/get_server_load: rate limited'], 429);
        }
        dfehc_set_transient_noautoload($rl_key, $cnt + 1, $window);
    }
    $load = dfehc_get_or_calculate_server_load();
    if ($load === false) {
        wp_send_json_success(dfehc_fallback_interval());
    }
    $interval = dfehc_calculate_recommended_interval_user_activity((float) $load);
    if ($interval <= 0) {
        $interval = dfehc_fallback_interval();
    }
    wp_send_json_success($interval);
}
dfehc_register_ajax('get_server_load', 'dfehc_get_server_load_ajax_handler');

function dfehc_calculate_server_load()
{
    if (function_exists('sys_getloadavg')) {
        $load = sys_getloadavg();
        if (isset($load[0])) {
            return (float) $load[0];
        }
    }
    if (is_readable('/proc/loadavg')) {
        $parts = explode(' ', (string) file_get_contents('/proc/loadavg'));
        if (isset($parts[0])) {
            return (float) $parts[0];
        }
    }
    $disabled = array_map('trim', explode(',', (string) ini_get('disable_functions')));
    if (function_exists('shell_exec') && !in_array('shell_exec', $disabled, true)) {
        $out = (string) (shell_exec('LANG=C uptime 2>/dev/null') ?? '');
        if ($out && preg_match('/load average[s]?:\s*([0-9.]+)/', $out, $m)) {
            return (float) $m[1];
        }
    }
    if (defined('DFEHC_PLUGIN_PATH')) {
        $estimator = rtrim((string) DFEHC_PLUGIN_PATH, "/\\") . '/defibrillator/load-estimator.php';
        if (file_exists($estimator)) {
            require_once $estimator;
            if (class_exists(\DynamicHeartbeat\Dfehc_ServerLoadEstimator::class)) {
                if (method_exists(\DynamicHeartbeat\Dfehc_ServerLoadEstimator::class, 'get_server_load')) {
                    $pct = \DynamicHeartbeat\Dfehc_ServerLoadEstimator::get_server_load();
                    if ($pct !== false && is_numeric($pct)) {
                        $cores = max(1, dfehc_get_cpu_cores());
                        return (float) ($cores * ((float) $pct) / 100.0);
                    }
                } elseif (method_exists(\DynamicHeartbeat\Dfehc_ServerLoadEstimator::class, 'estimate')) {
                    $raw = \DynamicHeartbeat\Dfehc_ServerLoadEstimator::estimate();
                    if (is_numeric($raw)) {
                        return (float) $raw;
                    }
                }
            }
        }
    }
    return false;
}

class Heartbeat_Async
{
    protected $action = 'dfehc_async_heartbeat';
    protected $scheduled = false;

    public function __construct()
    {
        add_action('init', [$this, 'maybe_schedule']);
        add_action($this->action, [$this, 'run_action']);
        dfehc_register_ajax($this->action, [$this, 'handle_async_request']);
    }

    public function maybe_schedule(): void
    {
        if ($this->scheduled === true) {
            return;
        }
        $this->scheduled = true;
        $interval = absint(300);
        $now = time();
        $aligned = $now - ($now % $interval) + $interval;
        $schedules = function_exists('wp_get_schedules') ? wp_get_schedules() : [];
        if (!isset($schedules['dfehc_5_minutes'])) {
            return;
        }
        if (!wp_next_scheduled($this->action)) {
            wp_schedule_event($aligned, 'dfehc_5_minutes', $this->action);
        } else {
            $next = wp_next_scheduled($this->action);
            if ($next === false || ($next - $now) > ($interval * 2)) {
                wp_schedule_single_event($now + $interval, $this->action);
            }
        }
    }

    public function handle_async_request(): void
    {
        $cap = apply_filters('dfehc_required_capability', DFEHC_CAPABILITY);
        $allow_public = (bool) apply_filters('dfehc_allow_public_async', false);
        if (!$allow_public) {
            $nonce_action = 'dfehc-' . $this->action;
            $valid = function_exists('check_ajax_referer')
                ? check_ajax_referer($nonce_action, 'nonce', false)
                : wp_verify_nonce(isset($_REQUEST['nonce']) ? (string) $_REQUEST['nonce'] : '', $nonce_action);
            if (!$valid) {
                wp_send_json_error(['message' => 'dfehc/dfehc_async_heartbeat: invalid nonce'], 403);
            }
            if (!current_user_can($cap)) {
                wp_send_json_error(['message' => 'dfehc/dfehc_async_heartbeat: not authorized'], 403);
            }
        } else {
            $limit = absint(apply_filters('dfehc_public_rate_limit', 30));
            $window = absint(apply_filters('dfehc_public_rate_window', 60));
            $ip = dfehc_client_ip();
            $rl_key = dfehc_scoped_key('dfehc_rl_' . md5($ip));
            $cnt = (int) get_transient($rl_key);
            if ($cnt >= $limit) {
                wp_send_json_error(['message' => 'dfehc/dfehc_async_heartbeat: rate limited'], 429);
            }
            dfehc_set_transient_noautoload($rl_key, $cnt + 1, $window);
        }
        try {
            $this->run_action();
            wp_send_json_success(true);
        } catch (\Throwable $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[dfehc] async error: ' . $e->getMessage());
            }
            wp_send_json_error(['message' => 'dfehc/dfehc_async_heartbeat: internal error'], 500);
        }
        wp_die();
    }

    protected function run_action(): void
    {
        $lock = dfehc_acquire_lock('dfehc_async_run', 30);
        if (!$lock) {
            return;
        }
        try {
            $last_activity = get_transient(dfehc_scoped_key('dfehc_last_user_activity'));
            if ($last_activity === false) {
                dfehc_set_transient_noautoload(dfehc_scoped_key(DFEHC_RECOMMENDED_INTERVAL), dfehc_fallback_interval(), 300);
                return;
            }
            $now = time();
            $elapsed = $now - (int) $last_activity;
            if ($elapsed < 0) {
                $elapsed = 0;
            }
            $load_raw = dfehc_calculate_server_load();
            if ($load_raw === false) {
                dfehc_set_transient_noautoload(dfehc_scoped_key(DFEHC_RECOMMENDED_INTERVAL), dfehc_fallback_interval(), 300);
                return;
            }
            $cores = max(1, dfehc_get_cpu_cores());
            $load_pct = min(100.0, round($load_raw / $cores * 100, 2));
            dfehc_set_transient_noautoload(dfehc_scoped_key(DFEHC_SERVER_LOAD), $load_pct, dfehc_server_load_ttl());
            $samples = get_transient(dfehc_scoped_key(DFEHC_LOAD_AVERAGES));
            if (!is_array($samples)) {
                $samples = [];
            }
            $samples[] = $load_pct;
            if (count($samples) > 5) {
                array_shift($samples);
            }
            dfehc_set_transient_noautoload(dfehc_scoped_key(DFEHC_LOAD_AVERAGES), $samples, 1800);
            $weights_raw = apply_filters('dfehc_load_weights', [5, 4, 3, 2, 1]);
            $wr = array_values((array) $weights_raw);
            if (!$wr) {
                $wr = [1];
            }
            $wr_count = count($wr);
            $weights = [];
            $n = count($samples);
            for ($i = 0; $i < $n; $i++) {
                $j = min($n - 1 - $i, $wr_count - 1);
                $w = isset($wr[$j]) ? $wr[$j] : 1;
                if (!is_numeric($w) || $w <= 0) {
                    $w = 1;
                }
                $weights[] = (float) $w;
            }
            $avg_load = dfehc_weighted_average($samples, $weights);
            $interval = $this->calculate_interval($elapsed, $avg_load);
            dfehc_set_transient_noautoload(dfehc_scoped_key(DFEHC_RECOMMENDED_INTERVAL), $interval, 300);
        } finally {
            dfehc_release_lock($lock);
        }
    }

    protected function calculate_interval(int $elapsed, float $load_pct): int
    {
        if ($elapsed <= dfehc_min_interval() && $load_pct < dfehc_max_server_load()) {
            return dfehc_min_interval();
        }
        if ($elapsed >= dfehc_max_interval()) {
            return dfehc_max_interval();
        }
        $load_factor = min(1.0, $load_pct / dfehc_max_server_load());
        $activity_factor = ($elapsed - dfehc_min_interval()) / (dfehc_max_interval() - dfehc_min_interval());
        $activity_factor = max(0.0, min(1.0, $activity_factor));
        $dominant = max($load_factor, $activity_factor);
        return (int) round(dfehc_min_interval() + $dominant * (dfehc_max_interval() - dfehc_min_interval()));
    }
}

function dfehc_weighted_average(array $values, array $weights): float
{
    if ($values === []) {
        return 0.0;
    }
    $tv = 0.0;
    $tw = 0.0;
    foreach ($values as $i => $v) {
        $w = isset($weights[$i]) ? $weights[$i] : 1;
        if (!is_numeric($w) || $w <= 0) {
            $w = 1;
        }
        $tv += (float) $v * (float) $w;
        $tw += (float) $w;
    }
    return $tw > 0 ? round($tv / $tw, 2) : 0.0;
}

function dfehc_calculate_recommended_interval_user_activity(float $current_load): int
{
    $key = dfehc_scoped_key(DFEHC_RECOMMENDED_INTERVAL);
    $interval = get_transient($key);
    if ($interval !== false) {
        return (int) $interval;
    }
    return $current_load >= dfehc_max_server_load() ? dfehc_max_interval() : dfehc_min_interval();
}

function dfehc_register_schedules(array $s): array
{
    if (!isset($s['dfehc_5_minutes'])) {
        $s['dfehc_5_minutes'] = ['interval' => 300, 'display' => __('Every 5 Minutes', 'dfehc')];
    }
    if (!isset($s['dfehc_daily'])) {
        $s['dfehc_daily'] = ['interval' => DAY_IN_SECONDS, 'display' => __('Once a day (DFEHC)', 'dfehc')];
    }
    return $s;
}
add_filter('cron_schedules', 'dfehc_register_schedules', 1);

function dfehc_prune_server_load_logs(): void
{
    $max_age = absint(apply_filters('dfehc_log_retention_seconds', DAY_IN_SECONDS));
    $max_cnt = absint(apply_filters('dfehc_log_retention_max', 1440));
    $now = time();
    $all_ids = function_exists('get_sites')
        ? array_map('intval', get_sites(['fields' => 'ids']))
        : [get_current_blog_id()];
    $chunk_size = absint(apply_filters('dfehc_prune_chunk_size', 50));
    $offset_key = 'dfehc_prune_offset';
    $offset = (int) get_site_option($offset_key, 0);
    $chunk = array_slice($all_ids, $offset, $chunk_size);
    if ($chunk === []) {
        $offset = 0;
        $chunk = array_slice($all_ids, 0, $chunk_size);
    }
    foreach ($chunk as $id) {
        $did_switch = false;
        if (is_multisite()) {
            switch_to_blog($id);
            $did_switch = true;
        }
        try {
            $option = 'dfehc_server_load_logs_' . get_current_blog_id();
            $logs = get_option($option, []);
            if ($logs) {
                $cutoff = $now - $max_age;
                $logs = array_filter(
                    $logs,
                    static function ($row) use ($cutoff) {
                        return isset($row['timestamp']) && (int) $row['timestamp'] >= $cutoff;
                    }
                );
                if (count($logs) > $max_cnt) {
                    $logs = array_slice($logs, -$max_cnt);
                }
                update_option($option, array_values($logs), false);
            }
        } finally {
            if ($did_switch) {
                restore_current_blog();
            }
        }
    }
    $offset += $chunk_size;
    update_site_option($offset_key, $offset);
}

add_action('dfehc_prune_logs_hook', 'dfehc_prune_server_load_logs');

if (!wp_next_scheduled('dfehc_prune_logs_hook')) {
    $t = strtotime('today 03:00');
    if ($t < time()) {
        $t += DAY_IN_SECONDS;
    }
    wp_schedule_event($t, 'dfehc_daily', 'dfehc_prune_logs_hook');
}

add_filter('dfehc_required_capability', function () { return 'manage_options'; });
add_filter('dfehc_server_load_ttl', function () { return 120; });
add_filter('dfehc_load_weights', function () { return [3, 2, 1]; });
add_filter('dfehc_async_retry', function () { return 1; });
add_filter('dfehc_log_retention_seconds', function () { return 2 * DAY_IN_SECONDS; });
add_filter('dfehc_log_retention_max', function () { return 3000; });
add_filter('dfehc_allow_public_server_load', '__return_false');
add_filter('dfehc_allow_public_async', '__return_false');

new Heartbeat_Async();