<?php
/*
Plugin Name: Dynamic Front-End Heartbeat Control
Plugin URI: https://heartbeat.support
Description: An enhanced solution to optimize the performance of your WordPress website. Stabilize your website's load averages and enhance the browsing experience for visitors during high-traffic fluctuations. 
Version: 1.2.995
Author: Codeloghin
Author URI: https://codeloghin.com
License: GPL2
*/

declare(strict_types=1);

if (!defined('DFEHC_PLUGIN_PATH')) {
    define('DFEHC_PLUGIN_PATH', plugin_dir_path(__FILE__));
}
if (!defined('DFEHC_CACHE_GROUP')) {
    define('DFEHC_CACHE_GROUP', 'dfehc');
}
if (!defined('DFEHC_MIN_INTERVAL')) {
    define('DFEHC_MIN_INTERVAL', 15);
}
if (!defined('DFEHC_MAX_INTERVAL')) {
    define('DFEHC_MAX_INTERVAL', 300);
}
if (!defined('DFEHC_MAX_SERVER_LOAD')) {
    define('DFEHC_MAX_SERVER_LOAD', 85);
}
if (!defined('DFEHC_MAX_RESPONSE_TIME')) {
    define('DFEHC_MAX_RESPONSE_TIME', 5000);
}
if (!defined('DFEHC_BATCH_SIZE')) {
    define('DFEHC_BATCH_SIZE', 75);
}
if (!defined('DFEHC_NONCE_ACTION')) {
    define('DFEHC_NONCE_ACTION', 'dfehc_get_recommended_intervals');
}
if (!defined('DFEHC_LOCK_TTL')) {
    define('DFEHC_LOCK_TTL', 60);
}
if (!defined('DFEHC_USER_ACTIVITY_TTL')) {
    define('DFEHC_USER_ACTIVITY_TTL', HOUR_IN_SECONDS);
}

require_once DFEHC_PLUGIN_PATH . 'engine/interval-helper.php';
require_once DFEHC_PLUGIN_PATH . 'engine/server-load.php';
require_once DFEHC_PLUGIN_PATH . 'engine/server-response.php';
require_once DFEHC_PLUGIN_PATH . 'engine/system-load-fallback.php';
require_once DFEHC_PLUGIN_PATH . 'visitor/manager.php';
require_once DFEHC_PLUGIN_PATH . 'visitor/cookie-helper.php';
require_once DFEHC_PLUGIN_PATH . 'defibrillator/unclogger.php';
require_once DFEHC_PLUGIN_PATH . 'defibrillator/rest-api.php';
require_once DFEHC_PLUGIN_PATH . 'defibrillator/db-health.php';
require_once DFEHC_PLUGIN_PATH . 'widget.php';
require_once DFEHC_PLUGIN_PATH . 'settings.php';

function dfehc_scoped_tkey(string $base): string
{
    if (function_exists('dfehc_scoped_key')) {
        return dfehc_scoped_key($base);
    }
    $bid = function_exists('get_current_blog_id') ? (string) get_current_blog_id() : '0';
    $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
    $salt = defined('DB_NAME') ? (string) DB_NAME : '';
    $tok = substr(md5((string) $host . $salt), 0, 10);
    return "{$base}_{$bid}_{$tok}";
}

function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void
{
    $jitter = function_exists('random_int') ? random_int(0, 5) : 0;
    $expiration = max(1, $expiration + $jitter);
    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
        wp_cache_set($key, $value, DFEHC_CACHE_GROUP, $expiration);
        return;
    }
    set_transient($key, $value, $expiration);
    if (isset($GLOBALS['wpdb'])) {
        global $wpdb;
        if (!isset($wpdb->options)) {
            return;
        }
        $opt_key_val = '_transient_' . $key;
        $opt_key_to  = '_transient_timeout_' . $key;
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_val));
        if ($autoload === 'yes') {
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->options} SET autoload='no' WHERE option_name=%s AND autoload='yes' LIMIT 1", $opt_key_val));
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->options} SET autoload='no' WHERE option_name=%s AND autoload='yes' LIMIT 1", $opt_key_to));
        }
        $wpdb->suppress_errors(false);
    }
}

function dfehc_enqueue_scripts(): void
{
    wp_register_script('dfehc-heartbeat', plugin_dir_url(__FILE__) . 'js/heartbeat.min.js', ['heartbeat'], '1.6.0', true);
    wp_enqueue_script('dfehc-heartbeat');

    $load = function_exists('dfehc_get_server_load') ? dfehc_get_server_load() : null;
    if ($load === false || $load === null) {
        $load = (float) DFEHC_MAX_SERVER_LOAD;
    }

    $recommended = function_exists('dfehc_calculate_recommended_interval_user_activity')
        ? dfehc_calculate_recommended_interval_user_activity((float) $load, DFEHC_BATCH_SIZE)
        : 60.0;

    $host = wp_parse_url(home_url(), PHP_URL_HOST) ?: 'site';
    $ver  = defined('DFEHC_VERSION') ? (string) DFEHC_VERSION : (string) filemtime(__FILE__);

    wp_localize_script('dfehc-heartbeat', 'dfehc_heartbeat_vars', [
        'recommendedInterval'       => (float) $recommended,
        'heartbeat_control_enabled' => get_option('dfehc_heartbeat_control_enabled', '1'),
        'nonce'                     => wp_create_nonce(DFEHC_NONCE_ACTION),
        'ver'                       => $ver,
        'cache_bypass_rate'         => 0.05,
    ]);
}
add_action('wp_enqueue_scripts', 'dfehc_enqueue_scripts');

function dfehc_get_user_activity_summary(int $batch_size = DFEHC_BATCH_SIZE): array
{
    $key = dfehc_scoped_tkey('dfehc_user_activity_summary');
    $cached = get_transient($key);
    if ($cached !== false && is_array($cached)) {
        return $cached;
    }
    $summary = dfehc_gather_user_activity_data($batch_size);
    dfehc_set_transient_noautoload($key, $summary, (int) DFEHC_USER_ACTIVITY_TTL);
    return $summary;
}

function dfehc_calculate_recommended_interval_user_activity(?float $load_average = null, int $batch_size = DFEHC_BATCH_SIZE): float
{
    if ($load_average === null) {
        $load_average = function_exists('dfehc_get_server_load') ? dfehc_get_server_load() : null;
    }
    if ($load_average === false || $load_average === null) {
        $load_average = (float) DFEHC_MAX_SERVER_LOAD;
    }
    $user_data = dfehc_get_user_activity_summary($batch_size);
    if (empty($user_data['total_weight'])) {
        return (float) DFEHC_MIN_INTERVAL;
    }
    $avg_duration = (float) $user_data['total_duration'] / max(1, (int) $user_data['total_weight']);
    return (float) dfehc_calculate_interval_based_on_duration($avg_duration, (float) $load_average);
}

function dfehc_gather_user_activity_data(int $batch_size): array
{
    $total_weighted_duration = 0.0;
    $total_weight            = 0;
    $offset                  = 0;
    while (true) {
        $userBatch = dfehc_get_users_in_batches($batch_size, $offset);
        if (!$userBatch) {
            break;
        }
        foreach ($userBatch as $user) {
            $activity = get_user_meta($user->ID, 'dfehc_user_activity', true);
            if (empty($activity['durations']) || !is_array($activity['durations'])) {
                continue;
            }
            $weight = count($activity['durations']);
            if ($weight <= 0) {
                continue;
            }
            $avg = array_sum($activity['durations']) / $weight;
            $total_weighted_duration += $weight * (float) $avg;
            $total_weight            += $weight;
        }
        $offset += $batch_size;
    }
    return ['total_duration' => $total_weighted_duration, 'total_weight' => $total_weight];
}

function dfehc_get_recommended_heartbeat_interval_async()
{
    if (!class_exists('Heartbeat_Async') && file_exists(__DIR__ . '/heartbeat-async.php')) {
        require_once __DIR__ . '/heartbeat-async.php';
    }
    if (!class_exists('Heartbeat_Async')) {
        $fallback = get_transient(dfehc_scoped_tkey('dfehc_recommended_interval'));
        return $fallback !== false ? (float) $fallback : (float) DFEHC_MIN_INTERVAL;
    }
    if (!class_exists('Dfehc_Get_Recommended_Heartbeat_Interval_Async')) {
        class Dfehc_Get_Recommended_Heartbeat_Interval_Async extends Heartbeat_Async
        {
            protected string $action = 'dfehc_get_recommended_interval_async';
            public function dispatch(): void
            {
                if (has_action($this->action)) {
                    do_action($this->action);
                } else {
                    if (!wp_next_scheduled($this->action)) {
                        wp_schedule_single_event(time() + 1, $this->action);
                    }
                }
            }
            protected function run_action(): void
            {
                $lock = function_exists('dfehc_acquire_lock') ? dfehc_acquire_lock('dfehc_interval_calculation_lock', (int) DFEHC_LOCK_TTL) : null;
                if (!$lock && function_exists('dfehc_acquire_lock')) {
                    return;
                }
                $last_key = dfehc_scoped_tkey('dfehc_last_user_activity');
                $ri_key   = dfehc_scoped_tkey('dfehc_recommended_interval');
                $last_activity = (int) get_transient($last_key);
                $elapsed       = max(0, time() - $last_activity);
                $load = function_exists('dfehc_get_server_load') ? dfehc_get_server_load() : null;
                if ($load === false || $load === null) {
                    $load = (float) DFEHC_MAX_SERVER_LOAD;
                }
                $interval = (float) dfehc_calculate_recommended_interval((float) $elapsed, (float) $load, 0.0);
                if ($interval <= 0) {
                    $interval = (float) DFEHC_MIN_INTERVAL;
                }
                dfehc_set_transient_noautoload($ri_key, $interval, 5 * MINUTE_IN_SECONDS);
                if ($lock && function_exists('dfehc_release_lock')) {
                    dfehc_release_lock($lock);
                }
            }
        }
    }

    $vis_key   = dfehc_scoped_tkey('dfehc_previous_visitor_count');
    $ri_key    = dfehc_scoped_tkey('dfehc_recommended_interval');

    $current = function_exists('dfehc_get_website_visitors') ? (int) dfehc_get_website_visitors() : 0;
    $prev    = get_transient($vis_key);
    $ratio   = (float) apply_filters('dfehc_visitors_delta_ratio', 0.2);
    if ($prev === false || ($current > 0 && abs($current - (int) $prev) > $current * $ratio)) {
        delete_transient($ri_key);
        dfehc_set_transient_noautoload($vis_key, $current, 5 * MINUTE_IN_SECONDS);
    }

    $cached = get_transient($ri_key);
    if ($cached !== false && is_numeric($cached)) {
        return (float) $cached;
    }

    $lock = function_exists('dfehc_acquire_lock') ? dfehc_acquire_lock('dfehc_interval_calculation_lock', (int) DFEHC_LOCK_TTL) : null;
    if ($lock || !function_exists('dfehc_acquire_lock')) {
        (new Dfehc_Get_Recommended_Heartbeat_Interval_Async())->dispatch();
        if ($lock && function_exists('dfehc_release_lock')) {
            dfehc_release_lock($lock);
        }
    }

    $val = get_transient($ri_key);
    if ($val === false || !is_numeric($val)) {
        return (float) dfehc_calculate_recommended_interval_user_activity();
    }
    return (float) $val;
}

function dfehc_get_recommended_intervals(): void
{
    check_ajax_referer(DFEHC_NONCE_ACTION, 'nonce');
    $ip = (string) ($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');
    $rlk = dfehc_scoped_tkey('dfehc_rl_' . md5($ip));
    $cnt = (int) get_transient($rlk);
    $limit = (int) apply_filters('dfehc_recommend_rl_limit', 30);
    $window = (int) apply_filters('dfehc_recommend_rl_window', 60);
    if ($cnt >= $limit) {
        wp_send_json_error(['message' => 'rate_limited'], 429);
    }
    dfehc_set_transient_noautoload($rlk, $cnt + 1, $window);
    $interval = dfehc_get_recommended_heartbeat_interval_async();
    if (!is_numeric($interval) || $interval <= 0) {
        $interval = (float) dfehc_calculate_recommended_interval_user_activity();
    }
    wp_send_json_success(['interval' => (float) $interval]);
}
add_action('wp_ajax_dfehc_update_heartbeat_interval', 'dfehc_get_recommended_intervals');
add_action('wp_ajax_nopriv_dfehc_update_heartbeat_interval', 'dfehc_get_recommended_intervals');

function dfehc_override_heartbeat_interval(array $settings): array
{
    $interval             = (int) ($settings['interval'] ?? DFEHC_MIN_INTERVAL);
    $interval             = min(max($interval, (int) DFEHC_MIN_INTERVAL), (int) DFEHC_MAX_INTERVAL);
    $settings['interval'] = $interval;
    return $settings;
}
add_filter('heartbeat_settings', 'dfehc_override_heartbeat_interval');

function dfehc_get_server_health_status(float $load): string
{
    if (get_option('dfehc_disable_heartbeat')) {
        return 'Stopped';
    }
    if ($load < 14.95) {
        return 'Resting';
    }
    if ($load <= 33.0) {
        return 'Pacing';
    }
    if ($load <= 67.65) {
        return 'Under Load';
    }
    return 'Under Strain';
}

function dfehc_invalidate_heartbeat_cache(): void
{
    $visitors = function_exists('dfehc_get_website_visitors') ? (int) dfehc_get_website_visitors() : 0;
    if ($visitors > 100) {
        delete_transient(dfehc_scoped_tkey('dfehc_recommended_interval'));
    }
}
add_action('wp_logout', 'dfehc_invalidate_heartbeat_cache');