<?php
declare(strict_types=1);

function dfehc_get_bot_pattern(): string
{
    static $pattern = null;
    if ($pattern !== null) {
        return $pattern;
    }
    $sigs = (array) apply_filters('dfehc_bot_signatures', [
        'bot','crawl','crawler','slurp','spider','mediapartners','bingpreview',
        'yandex','duckduckbot','baiduspider','sogou','exabot',
        'facebot','facebookexternalhit','ia_archiver',
    ]);
    $tokens = array_map(
        static fn(string $s): string =>
            '(?<![A-Za-z0-9])' . preg_quote($s, '/') . '(?![A-Za-z0-9])',
        $sigs
    );
    return $pattern = '/(' . implode('|', $tokens) . ')/i';
}

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int
    {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}

if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string
    {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}

if (!function_exists('dfehc_scoped_key')) {
    function dfehc_scoped_key(string $base): string
    {
        return $base . '_' . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}

if (!function_exists('dfehc_set_transient_noautoload')) {
    function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void
    {
        if (wp_using_ext_object_cache()) {
            if (function_exists('wp_cache_add')) {
                if (!wp_cache_add($key, $value, defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc', $expiration)) {
                    wp_cache_set($key, $value, defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc', $expiration);
                }
            } else {
                wp_cache_set($key, $value, defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc', $expiration);
            }
            return;
        }
        set_transient($key, $value, $expiration);
        global $wpdb;
        if (!isset($wpdb->options)) {
            return;
        }
        $opt_key = "_transient_$key";
        $opt_key_to = "_transient_timeout_$key";
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
        if ($autoload === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $wpdb->suppress_errors(false);
    }
}

function dfehc_ip_in_cidr(string $ip, string $cidr): bool
{
    if (strpos($cidr, '/') === false) {
        return (bool) filter_var($ip, FILTER_VALIDATE_IP) && $ip === $cidr;
    }
    [$subnet, $mask] = explode('/', $cidr, 2);
    $mask = (int) $mask;

    $ip_bin  = @inet_pton($ip);
    $sub_bin = @inet_pton($subnet);
    if ($ip_bin === false || $sub_bin === false) {
        return false;
    }
    if (strlen($ip_bin) !== strlen($sub_bin)) {
        return false;
    }

    $len = strlen($ip_bin);
    $max_bits = $len * 8;
    if ($mask < 0 || $mask > $max_bits) {
        return false;
    }

    $bytes = intdiv($mask, 8);
    $bits  = $mask % 8;

    if ($bytes && substr($ip_bin, 0, $bytes) !== substr($sub_bin, 0, $bytes)) {
        return false;
    }
    if ($bits === 0) {
        return true;
    }
    $ip_byte  = ord($ip_bin[$bytes]);
    $sub_byte = ord($sub_bin[$bytes]);
    $mask_byte = (0xFF << (8 - $bits)) & 0xFF;
    return ($ip_byte & $mask_byte) === ($sub_byte & $mask_byte);
}

function dfehc_trusted_proxy_request(): bool
{
    $remote = $_SERVER['REMOTE_ADDR'] ?? '';
    if ($remote === '') return false;
    $trusted = (array) apply_filters('dfehc_trusted_proxies', []);
    foreach ($trusted as $cidr) {
        if (is_string($cidr) && dfehc_ip_in_cidr($remote, $cidr)) {
            return true;
        }
    }
    return false;
}

function dfehc_select_client_ip_from_xff(string $xff, array $trustedCidrs): ?string
{
    $candidates = array_filter(array_map('trim', explode(',', $xff)), 'strlen');
    $ipNonTrusted = null;
    foreach ($candidates as $ip) {
        $ip = preg_replace('/%[0-9A-Za-z.\-]+$/', '', $ip);
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            continue;
        }
        $isTrustedHop = false;
        foreach ($trustedCidrs as $cidr) {
            if (is_string($cidr) && dfehc_ip_in_cidr($ip, $cidr)) {
                $isTrustedHop = true;
                break;
            }
        }
        if ($isTrustedHop) {
            continue;
        }
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
            return $ip;
        }
        if ($ipNonTrusted === null) {
            $ipNonTrusted = $ip;
        }
    }
    if ($ipNonTrusted !== null) {
        return $ipNonTrusted;
    }
    $last = trim((string) end($candidates));
    return filter_var($last, FILTER_VALIDATE_IP) ? $last : null;
}

function dfehc_is_request_bot(): bool
{
    static $cached = null;
    if ($cached !== null) {
        return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
    }

    $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if ($ua === '' || preg_match(dfehc_get_bot_pattern(), $ua)) {
        $cached = true;
        return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
    }

    $accept = $_SERVER['HTTP_ACCEPT'] ?? '';
    $sec_ch = $_SERVER['HTTP_SEC_CH_UA'] ?? '';
    if (($accept === '' || stripos($accept, 'text/html') === false) && $sec_ch === '') {
        $cached = true;
        return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
    }

    $ip     = dfehc_client_ip();
    $ipKeyScoped  = dfehc_scoped_key('dfehc_bad_ip_') . md5($ip ?: 'none');
    $group  = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');

    if ($ip && function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_get')) {
        if (wp_cache_get($ipKeyScoped, $group)) {
            $cached = true;
            return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
        }
    } else {
        $tkey = $ipKeyScoped . '_t';
        if ($ip && get_transient($tkey)) {
            $cached = true;
            return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
        }
    }

    $cached = false;
    return (bool) apply_filters('dfehc_is_request_bot', $cached, $_SERVER);
}

function dfehc_should_set_cookie(): bool
{
    if (defined('DOING_CRON') && DOING_CRON) return false;
    if (defined('WP_CLI') && WP_CLI) return false;
    if (function_exists('is_admin') && is_admin()) return false;
    if (function_exists('wp_doing_ajax') && wp_doing_ajax()) return false;
    if (function_exists('wp_is_json_request') && wp_is_json_request()) return false;
    if (function_exists('rest_get_url_prefix')) {
        $p = rest_get_url_prefix();
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        if ($p && strpos($uri, '/' . trim($p, '/') . '/') === 0) return false;
    }
    if (function_exists('is_feed') && is_feed()) return false;
    if (function_exists('is_robots') && is_robots()) return false;
    if (!empty($_SERVER['REQUEST_METHOD']) && strtoupper((string) $_SERVER['REQUEST_METHOD']) !== 'GET') return false;
    return true;
}

function dfehc_set_user_cookie(): void
{
    if (!dfehc_should_set_cookie()) {
        return;
    }
    if (dfehc_is_request_bot()) {
        return;
    }

    $ip     = dfehc_client_ip();
    $group  = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
    $maxRPM = (int) apply_filters('dfehc_max_rpm', 120);
    $badIpTtl = (int) apply_filters('dfehc_bad_ip_ttl', HOUR_IN_SECONDS);
    $scopedVisitorKey = dfehc_scoped_key('dfehc_total_visitors');

    if ($ip && function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_incr')) {
        $rpmKey = dfehc_scoped_key('dfehc_iprpm_') . md5($ip);
        $rpmTtl = 60 + (function_exists('random_int') ? random_int(0, 5) : 0);
        if (false === wp_cache_add($rpmKey, 0, $group, $rpmTtl)) {
            wp_cache_set($rpmKey, (int) (wp_cache_get($rpmKey, $group) ?: 0), $group, $rpmTtl);
        }
        $rpm = wp_cache_incr($rpmKey, 1, $group);
        if ($rpm === false) {
            wp_cache_set($rpmKey, 1, $group, $rpmTtl);
            $rpm = 1;
        } else {
            wp_cache_set($rpmKey, (int) $rpm, $group, $rpmTtl);
        }
        if ($rpm > $maxRPM) {
            $badKey = dfehc_scoped_key('dfehc_bad_ip_') . md5($ip);
            wp_cache_set($badKey, 1, $group, $badIpTtl);
            $tkey = $badKey . '_t';
            dfehc_set_transient_noautoload($tkey, 1, $badIpTtl);
            wp_cache_delete($rpmKey, $group);
            return;
        }
    }

    $name     = (string) apply_filters('dfehc_cookie_name', 'dfehc_user');
    $lifetime = (int) apply_filters('dfehc_cookie_lifetime', 400);
    $path     = (string) apply_filters('dfehc_cookie_path', defined('COOKIEPATH') ? COOKIEPATH : '/');
    $host     = function_exists('home_url') ? parse_url(home_url(), PHP_URL_HOST) : '';
    $isIpHost = $host && (filter_var($host, FILTER_VALIDATE_IP) !== false);
    $domainDefault = $isIpHost ? '' : ($host ?: (defined('COOKIE_DOMAIN') ? COOKIE_DOMAIN : ''));
    $domain   = (string) apply_filters('dfehc_cookie_domain', $domainDefault);

    $sameSite = (string) apply_filters('dfehc_cookie_samesite', 'Lax');
    $map = ['lax' => 'Lax', 'strict' => 'Strict', 'none' => 'None'];
    $sameSiteUpper = $map[strtolower($sameSite)] ?? 'Lax';

    $secure   = (function_exists('is_ssl') && is_ssl()) || $sameSiteUpper === 'None';
    if ($sameSiteUpper === 'None' && !$secure) {
        $sameSiteUpper = 'Lax';
    }
    $httpOnly = true;

    $nowTs = (int) ($_SERVER['REQUEST_TIME'] ?? time());
    $existing = $_COOKIE[$name] ?? '';
    $val = $existing;
    if (!preg_match('/^[A-Fa-f0-9]{32,64}$/', (string) $val)) {
        if (function_exists('random_bytes')) {
            $val = bin2hex(random_bytes(16));
        } else {
            $val = bin2hex(pack('H*', substr(sha1(uniqid((string) mt_rand(), true)), 0, 32)));
        }
    }

    $shouldRefresh = !isset($_COOKIE[$name]) || (mt_rand(0, 99) < (int) apply_filters('dfehc_cookie_refresh_percent', 5));
    if ($shouldRefresh) {
        if (!headers_sent()) {
            $expires = $nowTs + $lifetime;
            if (PHP_VERSION_ID >= 70300) {
                setcookie($name, $val, [
                    'expires'  => $expires,
                    'path'     => $path,
                    'domain'   => $domain ?: null,
                    'secure'   => $secure,
                    'httponly' => $httpOnly,
                    'samesite' => $sameSiteUpper,
                ]);
            } else {
                header(sprintf(
                    'Set-Cookie: %s=%s; Expires=%s; Path=%s%s%s; HttpOnly; SameSite=%s',
                    rawurlencode($name),
                    rawurlencode($val),
                    gmdate('D, d-M-Y H:i:s T', $expires),
                    $path,
                    $domain ? '; Domain=' . $domain : '',
                    $secure ? '; Secure' : '',
                    $sameSiteUpper
                ), false);
            }
        }
    }

    if (isset($_COOKIE[$name])) {
        return;
    }

    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_incr')) {
        $vTtl = $lifetime + (function_exists('random_int') ? random_int(0, 5) : 0);
        if (false === wp_cache_add($scopedVisitorKey, 0, $group, $vTtl)) {
            wp_cache_set($scopedVisitorKey, (int) (wp_cache_get($scopedVisitorKey, $group) ?: 0), $group, $vTtl);
        }
        $valInc = wp_cache_incr($scopedVisitorKey, 1, $group);
        if ($valInc === false) {
            wp_cache_set($scopedVisitorKey, 1, $group, $vTtl);
        } else {
            wp_cache_set($scopedVisitorKey, (int) $valInc, $group, $vTtl);
        }
        return;
    }

    $allowDirectClients = (bool) apply_filters('dfehc_enable_direct_cache_clients', false);

    static $client = null;
    if ($allowDirectClients && !$client && extension_loaded('redis') && class_exists('Redis')) {
        try {
            $client = new \Redis();
            $sock = function_exists('get_option') ? get_option('dfehc_redis_socket', '') : '';
            $ok   = $sock
                ? $client->pconnect($sock)
                : $client->pconnect(
                    function_exists('dfehc_get_redis_server') ? dfehc_get_redis_server() : '127.0.0.1',
                    function_exists('dfehc_get_redis_port')   ? dfehc_get_redis_port()   : 6379,
                    1.0
                );
            if ($ok) {
                $pass = apply_filters('dfehc_redis_auth', getenv('REDIS_PASSWORD') ?: null);
                $user = apply_filters('dfehc_redis_user', getenv('REDIS_USERNAME') ?: null);
                if ($user && $pass && method_exists($client, 'auth')) {
                    $client->auth([$user, $pass]);
                } elseif ($pass && method_exists($client, 'auth')) {
                    $client->auth($pass);
                }
                $pong = $client->ping();
                if (!in_array($pong, ['+PONG','PONG',true], true)) {
                    $client = null;
                }
            } else {
                $client = null;
            }
        } catch (\Throwable $e) {
            $client = null;
        }
    }
    if ($client) {
        $client->incr($scopedVisitorKey);
        $client->expire($scopedVisitorKey, $lifetime);
        return;
    }

    static $mem = null;
    if ($allowDirectClients && !$mem && extension_loaded('memcached') && class_exists('Memcached')) {
        $mem = new \Memcached('dfehc-cookie');
        if (!$mem->getServerList()) {
            $mem->addServer(
                function_exists('dfehc_get_memcached_server') ? dfehc_get_memcached_server() : '127.0.0.1',
                function_exists('dfehc_get_memcached_port') ? dfehc_get_memcached_port() : 11211
            );
        }
        if (empty($mem->getStats())) {
            $mem = null;
        }
    }
    if ($mem) {
        $valInc = $mem->increment($scopedVisitorKey, 1);
        if ($valInc === false) {
            $mem->set($scopedVisitorKey, 1, $lifetime);
        } else {
            $mem->touch($scopedVisitorKey, $lifetime);
        }
        return;
    }

    $cnt = (int) get_transient($scopedVisitorKey);
    $vTtl = $lifetime + (function_exists('random_int') ? random_int(0, 5) : 0);
    dfehc_set_transient_noautoload($scopedVisitorKey, $cnt + 1, $vTtl);
}

add_action('send_headers', 'dfehc_set_user_cookie', 1);