<?php
declare(strict_types=1);

if (!function_exists('dfehc_blog_id')) {
    function dfehc_blog_id(): int {
        return function_exists('get_current_blog_id') ? (int) get_current_blog_id() : 0;
    }
}
if (!function_exists('dfehc_host_token')) {
    function dfehc_host_token(): string {
        static $t = '';
        if ($t !== '') return $t;
        $host = @php_uname('n') ?: (defined('WP_HOME') ? WP_HOME : (function_exists('home_url') ? home_url() : 'unknown'));
        $salt = defined('DB_NAME') ? (string) DB_NAME : '';
        return $t = substr(md5((string) $host . $salt), 0, 10);
    }
}
if (!function_exists('dfehc_scoped_key')) {
    function dfehc_scoped_key(string $base): string {
        return $base . '_' . dfehc_blog_id() . '_' . dfehc_host_token();
    }
}
if (!function_exists('dfehc_set_transient_noautoload')) {
    function dfehc_set_transient_noautoload(string $key, $value, int $expiration): void {
        $group = defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc';
        if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache()) {
            if (function_exists('wp_cache_add')) {
                if (!wp_cache_add($key, $value, $group, $expiration)) {
                    wp_cache_set($key, $value, $group, $expiration);
                }
            } else {
                wp_cache_set($key, $value, $group, $expiration);
            }
            return;
        }
        set_transient($key, $value, $expiration);
        global $wpdb;
        if (!isset($wpdb->options)) return;
        $opt_key = "_transient_$key";
        $opt_key_to = "_transient_timeout_$key";
        $wpdb->suppress_errors(true);
        $autoload = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key));
        if ($autoload === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $autoload_to = $wpdb->get_var($wpdb->prepare("SELECT autoload FROM {$wpdb->options} WHERE option_name=%s LIMIT 1", $opt_key_to));
        if ($autoload_to === 'yes') {
            $wpdb->update($wpdb->options, ['autoload' => 'no'], ['option_name' => $opt_key_to, 'autoload' => 'yes'], ['%s'], ['%s','%s']);
        }
        $wpdb->suppress_errors(false);
    }
}
if (!function_exists('dfehc_get_redis_server')) {
    function dfehc_get_redis_server(): string {
        $h = getenv('REDIS_HOST');
        return $h ? (string) $h : '127.0.0.1';
    }
}
if (!function_exists('dfehc_get_redis_port')) {
    function dfehc_get_redis_port(): int {
        $p = getenv('REDIS_PORT');
        return $p && ctype_digit((string) $p) ? (int) $p : 6379;
    }
}
if (!function_exists('dfehc_get_memcached_server')) {
    function dfehc_get_memcached_server(): string {
        $h = getenv('MEMCACHED_HOST');
        return $h ? (string) $h : '127.0.0.1';
    }
}
if (!function_exists('dfehc_get_memcached_port')) {
    function dfehc_get_memcached_port(): int {
        $p = getenv('MEMCACHED_PORT');
        return $p && ctype_digit((string) $p) ? (int) $p : 11211;
    }
}
if (!defined('DFEHC_SENTINEL_NO_LOAD')) {
    define('DFEHC_SENTINEL_NO_LOAD', -1);
}

if (!function_exists('dfehc_acquire_lock')) {
    function dfehc_acquire_lock(string $key, int $ttl = 60) {
        $group = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
        $scoped = dfehc_scoped_key($key);
        if (class_exists('WP_Lock')) {
            $lock = new WP_Lock($scoped, $ttl);
            return $lock->acquire() ? $lock : null;
        }
        if (function_exists('wp_cache_add') && wp_cache_add($scoped, 1, $group, $ttl)) {
            return (object) ['cache_key' => $scoped, 'cache_group' => $group];
        }
        if (false !== get_transient($scoped)) {
            return null;
        }
        if (set_transient($scoped, 1, $ttl)) {
            return (object) ['transient_key' => $scoped];
        }
        return null;
    }
    function dfehc_release_lock($lock): void {
        if ($lock instanceof WP_Lock) {
            $lock->release();
            return;
        }
        if (is_object($lock) && isset($lock->cache_key)) {
            $group = $lock->cache_group ?? apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
            wp_cache_delete($lock->cache_key, $group);
            return;
        }
        if (is_object($lock) && isset($lock->transient_key)) {
            delete_transient($lock->transient_key);
        }
    }
}

function dfehc_set_default_last_activity_time(int $user_id): void {
    $meta_key = (string) apply_filters('dfehc_last_activity_meta_key', 'last_activity_time');
    update_user_meta($user_id, $meta_key, time());
}
add_action('user_register', 'dfehc_set_default_last_activity_time');

function dfehc_add_intervals(array $s): array {
    $s['dfehc_5_minutes'] ??= ['interval' => 300, 'display' => __('Every 5 minutes (DFEHC)', 'dfehc')];
    $s['dfehc_daily'] ??= ['interval' => DAY_IN_SECONDS, 'display' => __('Daily (DFEHC)', 'dfehc')];
    return $s;
}
add_filter('cron_schedules', 'dfehc_add_intervals');

function dfehc_schedule_user_activity_processing(): void {
    $lock = dfehc_acquire_lock('dfehc_cron_sched_lock', 15);
    $aligned = time() - time() % 300 + 300;
    if (!get_option('dfehc_activity_cron_scheduled') && !wp_next_scheduled('dfehc_process_user_activity')) {
        $ok = wp_schedule_event($aligned, 'dfehc_5_minutes', 'dfehc_process_user_activity');
        if ($ok) {
            update_option('dfehc_activity_cron_scheduled', 1, false);
        } else {
            wp_schedule_single_event($aligned, 'dfehc_process_user_activity');
        }
    }
    if (!wp_next_scheduled('dfehc_cleanup_user_activity')) {
        $args = [0, (int) apply_filters('dfehc_cleanup_batch_size', 75)];
        $ok2 = wp_schedule_event($aligned + 300, 'dfehc_daily', 'dfehc_cleanup_user_activity', $args);
        if (!$ok2) {
            wp_schedule_single_event($aligned + 300, 'dfehc_cleanup_user_activity', $args);
        }
    }
    if ($lock) {
        dfehc_release_lock($lock);
    }
}
add_action('init', 'dfehc_schedule_user_activity_processing');

function dfehc_throttled_user_activity_handler(): void {
    $lock = dfehc_acquire_lock('dfehc_recent_user_processing', 300);
    if (!$lock) {
        return;
    }

	$prev = (bool) ignore_user_abort(true);

    try {
        dfehc_process_user_activity();
    } finally {
        ignore_user_abort((bool) $prev);
        dfehc_release_lock($lock);
    }
}
add_action('dfehc_process_user_activity', 'dfehc_throttled_user_activity_handler');

function dfehc_process_user_activity(): void {
    $flag_opt = dfehc_scoped_key('dfehc_activity_backfill_done');
    if (get_option($flag_opt)) {
        return;
    }
    global $wpdb;
    $meta_key = (string) apply_filters('dfehc_last_activity_meta_key', 'last_activity_time');
    $batch = (int) apply_filters('dfehc_activity_processing_batch_size', 75);
    $batch = max(1, min(500, $batch));
    $last_id_opt = dfehc_scoped_key('dfehc_activity_last_id');
    $last_id = (int) get_option($last_id_opt, 0);
    $ids = $wpdb->get_col($wpdb->prepare(
        "SELECT ID FROM $wpdb->users WHERE ID > %d ORDER BY ID ASC LIMIT %d",
        $last_id, $batch
    ));
    if (!$ids) {
        update_option($flag_opt, 1, false);
        delete_option($last_id_opt);
        update_option(dfehc_scoped_key('dfehc_last_activity_cron'), time(), false);
        return;
    }
    $now = time();
    $written = 0;
    $max_writes = (int) apply_filters('dfehc_activity_max_writes_per_run', 500);
    foreach ($ids as $id) {
        if (!get_user_meta($id, $meta_key, true)) {
            update_user_meta($id, $meta_key, $now);
            $written++;
            if ($written >= $max_writes) {
                break;
            }
        }
    }
    update_option($last_id_opt, end($ids), false);
    update_option(dfehc_scoped_key('dfehc_last_activity_cron'), time(), false);
}

function dfehc_record_user_activity(): void {
    if (!function_exists('is_user_logged_in') || !is_user_logged_in()) {
        return;
    }
    static $cache = [];
    $meta_key = (string) apply_filters('dfehc_last_activity_meta_key', 'last_activity_time');
    $uid = get_current_user_id();
    $now = time();
    $interval = (int) apply_filters('dfehc_activity_update_interval', 900);
    $interval = max(60, $interval);
    $last = $cache[$uid] ?? (int) get_user_meta($uid, $meta_key, true);
    if ($now - $last >= $interval) {
        update_user_meta($uid, $meta_key, $now);
        $cache[$uid] = $now;
    }
}
add_action('wp', 'dfehc_record_user_activity');

function dfehc_cleanup_user_activity(int $last_id = 0, int $batch_size = 75): void {
    $lock = dfehc_acquire_lock('dfehc_cleanup_lock', 600);
    if (!$lock) {
        return;
    }
    $prev = ignore_user_abort(true);
    if (function_exists('set_time_limit')) {
        @set_time_limit(30);
    }
    try {
        global $wpdb;
        $meta_key = (string) apply_filters('dfehc_last_activity_meta_key', 'last_activity_time');
        $batch_size = (int) apply_filters('dfehc_cleanup_batch_size', $batch_size);
        $batch_size = max(1, min(500, $batch_size));
        $ids = $wpdb->get_col($wpdb->prepare(
            "SELECT ID FROM $wpdb->users WHERE ID > %d ORDER BY ID ASC LIMIT %d",
            $last_id, $batch_size
        ));
        if (!$ids) {
            update_option(dfehc_scoped_key('dfehc_last_cleanup_cron'), time(), false);
            return;
        }
        $cutoff = time() - (int) apply_filters('dfehc_activity_expiration', WEEK_IN_SECONDS);
        foreach ($ids as $id) {
            $ts = (int) get_user_meta($id, $meta_key, true);
            if ($ts && $ts < $cutoff) {
                delete_user_meta($id, $meta_key);
            }
        }
        if (count($ids) === $batch_size) {
            wp_schedule_single_event(time() + 15 + (function_exists('random_int') ? random_int(0, 5) : rand(0, 5)), 'dfehc_cleanup_user_activity', [end($ids), $batch_size]);
        }
        update_option(dfehc_scoped_key('dfehc_last_cleanup_cron'), time(), false);
    } finally {
        ignore_user_abort($prev);
        dfehc_release_lock($lock);
    }
}
add_action('dfehc_cleanup_user_activity', 'dfehc_cleanup_user_activity', 10, 2);

function dfehc_increment_total_visitors(): void {
    $key = dfehc_scoped_key('dfehc_total_visitors');
    $grp = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
    $ttl = (int) apply_filters('dfehc_total_visitors_ttl', HOUR_IN_SECONDS);
    $ttl += function_exists('random_int') ? random_int(0, 5) : 0;

    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_incr')) {
        if (false === wp_cache_add($key, 0, $grp, $ttl)) {
            $existing = wp_cache_get($key, $grp);
            wp_cache_set($key, (int) ($existing ?: 0), $grp, $ttl);
        }
        $val = wp_cache_incr($key, 1, $grp);
        if ($val === false) {
            wp_cache_set($key, 1, $grp, $ttl);
        }
        return;
    }

    $allowDirectClients = (bool) apply_filters('dfehc_enable_direct_cache_clients', false);

    static $redis = null;
    if ($allowDirectClients && !$redis && extension_loaded('redis') && class_exists('Redis')) {
        try {
            $redis = new \Redis();
            $ok = $redis->pconnect(dfehc_get_redis_server(), dfehc_get_redis_port(), 1.0);
            if ($ok) {
                $pass = apply_filters('dfehc_redis_auth', getenv('REDIS_PASSWORD') ?: null);
                $user = apply_filters('dfehc_redis_user', getenv('REDIS_USERNAME') ?: null);
                if ($user && $pass && method_exists($redis, 'auth')) {
                    $redis->auth([$user, $pass]);
                } elseif ($pass && method_exists($redis, 'auth')) {
                    $redis->auth($pass);
                }
                $pong = $redis->ping();
                if (!in_array($pong, ['+PONG','PONG', true], true)) {
                    $redis = null;
                }
            } else {
                $redis = null;
            }
        } catch (\Throwable $e) {
            $redis = null;
        }
    }
    if ($redis) {
        $redis->incr($key);
        $redis->expire($key, $ttl);
        return;
    }

    static $mem = null;
    if ($allowDirectClients && !$mem && extension_loaded('memcached') && class_exists('Memcached')) {
        $mem = new \Memcached('dfehc-visitors');
        if (!$mem->getServerList()) {
            $mem->addServer(dfehc_get_memcached_server(), dfehc_get_memcached_port());
        }
        if (empty($mem->getStats())) {
            $mem = null;
        }
    }
    if ($mem) {
        $inc = $mem->increment($key, 1);
        if ($inc === false) {
            $mem->set($key, 1, $ttl);
        } else {
            $mem->touch($key, $ttl);
        }
        return;
    }

    $cnt = (int) get_transient($key);
    if ($cnt === 0) {
        dfehc_set_transient_noautoload($key, 0, $ttl);
        $cnt = 0;
    }
    dfehc_set_transient_noautoload($key, $cnt + 1, $ttl);
}
function dfehc_increment_total_visitors_fallback(): void {
    dfehc_increment_total_visitors();
}

function dfehc_safe_cache_get(string $key): int {
    $scoped = dfehc_scoped_key($key);
    $grp = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_get')) {
        $v = wp_cache_get($scoped, $grp);
        return is_numeric($v) ? (int) $v : 0;
    }
    $v = get_transient($scoped);
    return is_numeric($v) ? (int) $v : 0;
}

function dfehc_safe_cache_delete(string $key): void {
    $scoped = dfehc_scoped_key($key);
    $grp = apply_filters('dfehc_cache_group', defined('DFEHC_CACHE_GROUP') ? DFEHC_CACHE_GROUP : 'dfehc');
    if (function_exists('wp_using_ext_object_cache') && wp_using_ext_object_cache() && function_exists('wp_cache_delete')) {
        wp_cache_delete($scoped, $grp);
    }
    delete_transient($scoped);
}

function dfehc_get_website_visitors(): int {
    $cache_key = dfehc_scoped_key('dfehc_total_visitors_cache');
    $regen_key = dfehc_scoped_key('dfehc_regenerating_cache');
    $stale_opt = dfehc_scoped_key('dfehc_stale_total_visitors');
    $cache = get_transient($cache_key);
    if ($cache !== false) {
        return (int) $cache;
    }
    if (get_transient($regen_key)) {
        $stale = get_option($stale_opt, 0);
        return is_numeric($stale) ? (int) $stale : 0;
    }

    $regen_ttl = MINUTE_IN_SECONDS + (function_exists('random_int') ? random_int(0, 5) : 0);
    dfehc_set_transient_noautoload($regen_key, true, $regen_ttl);
    $total = dfehc_safe_cache_get('dfehc_total_visitors');
    $ttl = (int) apply_filters('dfehc_visitors_cache_ttl', 10 * MINUTE_IN_SECONDS);
    $ttl += function_exists('random_int') ? random_int(0, 5) : 0;
    dfehc_set_transient_noautoload($cache_key, (int) $total, $ttl);
    update_option($stale_opt, (int) $total, false);
    delete_transient($regen_key);

    return (int) apply_filters('dfehc_get_website_visitors_result', (int) $total);
}

function dfehc_get_users_in_batches(int $batch_size, int $offset): array {
    $batch_size = max(1, min(1000, $batch_size));
    $offset = max(0, $offset);
    $query = new WP_User_Query([
        'number' => $batch_size,
        'offset' => $offset,
        'fields' => ['ID'],
    ]);
    $res = $query->get_results();
    return is_array($res) ? $res : [];
}

function dfehc_reset_total_visitors(): void {
    $start = microtime(true);
    $lock = dfehc_acquire_lock('dfehc_resetting_visitors', 60);
    if (!$lock) {
        return;
    }
    try {
        $threshold = (float) apply_filters('dfehc_reset_load_threshold', 15.0);
        $load = function_exists('dfehc_get_server_load') ? dfehc_get_server_load() : DFEHC_SENTINEL_NO_LOAD;
        if (!is_numeric($load)) {
            return;
        }
        $load = (float) $load;
        if ($load === (float) DFEHC_SENTINEL_NO_LOAD || $load >= $threshold) {
            return;
        }
        dfehc_safe_cache_delete('dfehc_total_visitors');
        delete_option(dfehc_scoped_key('dfehc_stale_total_visitors'));
        delete_transient(dfehc_scoped_key('dfehc_total_visitors'));
        delete_transient(dfehc_scoped_key('dfehc_regenerating_cache'));
        if (microtime(true) - $start > 5) {
            return;
        }
    } finally {
        dfehc_release_lock($lock);
    }
}
add_action('dfehc_reset_total_visitors_event', 'dfehc_reset_total_visitors');

function dfehc_on_activate(): void {
    $aligned = time() - time() % 300 + 300;
    if (!wp_next_scheduled('dfehc_reset_total_visitors_event')) {
        $ok = wp_schedule_event($aligned + HOUR_IN_SECONDS, 'hourly', 'dfehc_reset_total_visitors_event');
        if (!$ok) {
            wp_schedule_single_event($aligned + HOUR_IN_SECONDS, 'dfehc_reset_total_visitors_event');
        }
    }
    dfehc_process_user_activity();
}
if (function_exists('register_activation_hook')) {
    register_activation_hook(__FILE__, 'dfehc_on_activate');
}

function dfehc_on_deactivate(): void {
    wp_clear_scheduled_hook('dfehc_process_user_activity');
    wp_clear_scheduled_hook('dfehc_reset_total_visitors_event');
    wp_clear_scheduled_hook('dfehc_cleanup_user_activity');
    delete_option('dfehc_activity_cron_scheduled');
    delete_option(dfehc_scoped_key('dfehc_activity_backfill_done'));
}
if (function_exists('register_deactivation_hook')) {
    register_deactivation_hook(__FILE__, 'dfehc_on_deactivate');
}

if (defined('WP_CLI') && WP_CLI) {
    \WP_CLI::add_command('dfehc:reset_visitors', static function () {
        dfehc_reset_total_visitors();
        \WP_CLI::success('Visitor count reset triggered.');
    });
}